using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Model;
using UnityEngine.Networking;

namespace Magify
{
    internal partial class ServerApi
    {
        public bool IsContextRequest => _requestContextContext.CheckSingleRequestInProgress();

        public UniTask<CampaignsContext> GetContextAsync(ConfigScope scopes, CancellationToken cancellationToken)
        {
            ThrowIfDisposed();
            var config = new RequestConfig(
                WebRequestMethods.GetContext,
                createGetContextRequestPayload,
                useWebRequestWithRepeats: true,
                isAuthTokenRequired: true);
            var request = _interactionsProvider.MakeAsyncRequest(
                in _requestContextContext,
                in config,
                cancellationToken
            );
            return request.HandleByDefault(config, rejectedResult: null);

            WebRequestMessage createGetContextRequestPayload()
            {
                ThrowIfDisposed();
                var disabledCampaignTypes = _generalPrefs.UsedDefaultCampaignTypes;
                var source = _generalPrefs.MediaSource;
                return new WebRequestMessage
                {
                    Method = WebRequestMethods.GetContext,
                    Payload = new ContextPayload
                    {
                        DisabledCampaignTypes = disabledCampaignTypes,
                        NetworkName = source.NetworkName,
                        CampaignName = source.CampaignName,
                        GroupName = source.GroupName,
                        Scopes = scopes.ToStringArray(),
                    },
                };
            }
        }

        private RepeatState GetContextResponseHandler(WebResponseMessage response, out CampaignsContext result)
        {
            if (response.Result == UnityWebRequest.Result.Success && !string.IsNullOrEmpty(response.Text))
            {
                _logger.Log($"Data loaded successfully. Result text={response.Text}");
                if (ServerApiUtils.TryDeserializeFromBasicResponse(response.Text, _logger, out result, out var exception))
                {
                    return result == null ? RepeatState.Wait : RepeatState.Finish;
                }
                else
                {
                    _logger.LogWarning($"Failed to parse context response: {exception.Message}");
                    result = default;
                    return RepeatState.Wait;
                }
            }

            if (!string.IsNullOrEmpty(response.Text))
            {
                var errorResponse = ServerApiUtils.TryExtractErrorResponse(response.Text, _logger);
                if (ServerApiUtils.TryHandleDefaultErrorCodes(errorResponse?.Error, response.RequestMessage.Method, AuthorizationToken, _logger, HandleBannedResponse, out var repeatState))
                {
                    result = default;
                    return repeatState;
                }
            }

            result = default;
            return RepeatState.Wait;
        }

        public UniTask CancelAllContextLoadings()
        {
            _logger.Log($"{nameof(CancelAllContextLoadings)} has been called");
            _requestContextContext.CancelAllRequests();
            var promise = _requestContextContext.SingleModeRequestTools.CompletionSourceRaw;
            if (promise == null || promise.IsCompleted())
            {
                return UniTask.CompletedTask;
            }
            return promise.Task.SuppressCancellationThrow();
        }
    }
}