using System;
using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using Magify.Model;
using UnityEngine;

namespace Magify
{
    internal readonly struct ParsedCampaign
    {
        [NotNull]
        public readonly ICampaign Campaign;

        [CanBeNull]
        public readonly AbstractProductsCollection Products;

        public ParsedCampaign([NotNull] ICampaign campaign, [CanBeNull] AbstractProductsCollection products = null)
        {
            Campaign = campaign;
            Products = products;
        }
    }

    internal abstract class CampaignParser
    {
        protected static readonly MagifyLogger Logger = MagifyLogger.Get();

        internal readonly struct AdditionalData
        {
            [NotNull]
            public string ClientId { get; }

            public AdditionalData([NotNull] string clientId)
            {
                ClientId = clientId;
            }
        }

        internal static readonly Dictionary<ProductType, ProductParser> ProductsParsers = new()
        {
            { ProductType.Bonus, new BonusProductParser() },
            { ProductType.Rewarded, new RewardProductParser() },
            { ProductType.Consumable, new InAppProductParser() },
            { ProductType.NonConsumable, new InAppProductParser() },
            { ProductType.Subscription, new SubscriptionProductParser() },
            { ProductType.Info, new InfoProductParser() },
            { ProductType.CrossPromo, new CrossPromoProductParser() },
            { ProductType.ExternalLink, new ExternalLinkProductParser() },
            { ProductType.InternalLink, new InternalLinkProductParser() },
        };

        internal static readonly Dictionary<CreativeType, CreativeParser> CreativeParsers = new()
        {
            { CreativeType.Hardcoded, new CustomCreativeParser() },
            { CreativeType.Screen, new CustomCreativeParser() },
            { CreativeType.Image, new ImageCreativeParser() },
            { CreativeType.SystemAlert, new SystemAlertCreativeParser() },
            { CreativeType.Playable, new PlayableCreativeParser() },
            { CreativeType.NotificationAlert, new NotificationAlertCreativeParser() },
            { CreativeType.Bundle, new BundleCreativeParser() },
            { CreativeType.NoUI, new NoUiCreativeParser() },
        };

        internal static readonly Dictionary<SplashType, SplashScreenParser> SplashScreenParsers = new()
        {
            { SplashType.NoSplash, new NoSplashScreenParser() },
            { SplashType.Loader, new LoaderSplashScreenParser() },
            { SplashType.Popup, new PopupSplashScreenParser() },
        };

        [NotNull]
        public static CampaignInfo ExtractCampaignInfo([NotNull] CampaignModel model)
        {
            var purchaseLimit = new PurchaseLimits
            {
                ExcludeIfAll = model.ExcludeIfAllPurchased?.Select(item => item.Product).ToHashSet(),
                ExcludeIfAny = model.ExcludeIfAnyPurchased?.Select(item => item.Product).ToHashSet(),
                IncludeIfAll = model.IncludeIfAllPurchased?.Select(item => item.Product).ToHashSet(),
                IncludeIfAny = model.IncludeIfAnyPurchased?.Select(item => item.Product).ToHashSet()
            };

            var referrerLimit = new ReferrerLimits
            {
                IncludeIds = model.IncludeReferrerIds?.Select(item => item.Id).ToHashSet(),
                ExcludeIds = model.ExcludeReferrerIds?.Select(item => item.Id).ToHashSet()
            };

            var rewardGrantLimit = new GrantLimits
            {
                Global = model.GlobalRewardLimit,
                Session = model.SessionRewardLimit,
                Daily = model.DailyRewardLimit
            };

            var bonusGrantLimit = new GrantLimits
            {
                Global = model.GlobalBonusLimit,
                Session = model.SessionBonusLimit,
                Daily = model.DailyBonusLimit
            };

            return new CampaignInfo
            {
                Name = model.Name,
                Type = model.Type!.Value,
                SubscriptionStatus = model.SubscriptionStatus ?? CampaignSubscriptionStatus.AllUsers,
                InAppStatus = model.InAppStatus ?? CampaignInAppStatus.AllUsers,
                AuthorizationStatus = model.AuthorizationStatus ?? CampaignAuthorizationStatus.AllUsers,
                PayingStatus = model.PayingStatus ?? CampaignPayingStatus.AllUsers,
                Events = model.Events,
                Placements = model.Placements,
                ImpressionLimits = model.ImpressionLimit,
                ClickLimit = model.ClickLimit,
                PurchaseLimits = purchaseLimit,
                ReferrerLimits = referrerLimit,
                RewardGrantLimits = rewardGrantLimit,
                BonusGrantLimits = bonusGrantLimit
            };
        }

        public bool IsValid(CampaignModel model)
        {
            var isValidModel = IsValidModel(model);
            if (isValidModel && model.Type!.Value.IsSupportNested() is true)
            {
                return HasRequiredNested(model);
            }
            return isValidModel;
        }

        private static bool HasRequiredNested(CampaignModel model)
        {
            var campaignType = model.Type!.Value;
            if (model.NestedCampaigns == null)
            {
                Logger.LogWarning($"Unable to check a nested campaigns for {model.Name} ({campaignType}) - \"nested_campaigns\" collection is empty");
                return false;
            }

            foreach (var nestedCampaign in model.NestedCampaigns)
            {
                if (!nestedCampaign.Type.HasValue)
                {
                    Logger.LogWarning($"Unable to check a nested campaign with name {nestedCampaign.Name}, it has invalid {nameof(nestedCampaign.Type)} field.");
                }
                else if (ProductsParsers.TryGetValue(nestedCampaign.Type.Value, out var parser))
                {
                    var isValid = parser.IsValidModel(nestedCampaign);
                    if (isValid && campaignType.IsRequiredProductType(nestedCampaign.Type.Value) is true)
                    {
                        return true;
                    }
                }
                else
                {
                    Logger.LogWarning($"Unable to check a nested campaign with name {nestedCampaign.Name}, there is no parser for it");
                }
            }
            return false;
        }

        protected abstract bool IsValidModel(CampaignModel model);

        protected static bool IsValidLimitedOffer(CampaignModel model)
        {
            var period = model.ActivationPeriod;
            if (period == null)
            {
                return true;
            }
            if (period.StartDay == null || period.StartTime == null || period.EndDay == null || period.EndTime == null)
            {
                Logger.LogError($"Invalid Limited offer model: {model.Name}. {nameof(ActivationPeriod)} values should not be empty.");
                return false;
            }
            return true;
        }

        public abstract ParsedCampaign Parse([NotNull] CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data);
    }

    internal abstract class CampaignParser<T> : CampaignParser
        where T : ICampaign
    {
        protected static bool IsValidCampaignWithScreen([NotNull] CampaignModel model)
        {
            if (model.Info == null)
            {
                Logger.LogError($"Invalid model of {typeof(T).Name}: {model.Name}. MetaInfo was not found.");
                return false;
            }

            if (model.Info.Creative == null)
            {
                Logger.LogError($"Invalid model of {typeof(T).Name}: {model.Name}. Creative was not found.");
                return false;
            }

            if (model.Info.Creative.Type != null && CreativeParsers.TryGetValue(model.Info.Creative.Type.Value, out var parser))
            {
                return parser.IsValid(model.Info.Creative);
            }
            Logger.LogError($"Unexpected creative model: \n{JsonFacade.SerializeObject(model)}");
            return false;
        }

        protected ParsedCampaign Result<TProduct>(
            [NotNull] CampaignModel model,
            IEnumerable<NestedCampaign> nestedCampaigns,
            [NotNull] string clientId,
            Func<ICreative, IReadOnlyList<TProduct>, T> creator)
            where TProduct : ProductDef
        {
            var creative = ToCampaignCreative(model.Info!.Creative);
            var products = nestedCampaigns
                .Select(c => ProductsParsers[c.Type!.Value].ParseProduct(c, clientId) as TProduct)
                .Where(c => c != null)
                .ToList();
            var parsedProducts = new AbstractProductsCollection(products);
            return new ParsedCampaign(creator(creative, products), parsedProducts);
        }

        [NotNull]
        protected static ICreative ToCampaignCreative(CreativeModel model)
        {
            var attributes = model.Type switch
            {
                CreativeType.Hardcoded => model.Info?.Attributes,
                _ => model.Info?.CustomAttributes
            };

            return CreativeParsers[model.Type!.Value].Parse(model, attributes);
        }

        [NotNull]
        internal static ISplashScreen ToInterSplashScreen(SplashscreenModel model)
        {
            return SplashScreenParsers[model.Type!.Value].Parse(model);
        }
    }

    #region Realizations

    internal class BannerCampaignParser : CampaignParser<BannerCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (model.Info?.BannerPosition == null)
            {
                Logger.LogError($"Invalid model of {nameof(BannerCampaign)}: {model.Name}. Unknown type of {nameof(BannerPosition)}.");
                return false;
            }
            return true;
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return new ParsedCampaign(new BannerCampaign
            {
                Name = model.Name,
                Position = model.Info!.BannerPosition == Model.BannerPosition.Top
                    ? BannerPosition.Top
                    : BannerPosition.Bottom
            });
        }
    }

    internal class InterstitialCampaignParser : CampaignParser<InterstitialCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (model.Info?.Splash?.Type != null && SplashScreenParsers.TryGetValue(model.Info.Splash.Type.Value, out var parser))
            {
                return parser.IsValid(model.Info.Splash);
            }
            Logger.LogError($"Invalid model of {nameof(InterstitialCampaign)}: {model.Name}. Splashscreen was not found.");
            return false;
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return new ParsedCampaign(new InterstitialCampaign
            {
                Name = model.Name,
                Screen = ToInterSplashScreen(model.Info!.Splash)
            });
        }
    }

    internal class RewardedVideoCampaignParser : CampaignParser<RewardedVideoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<RewardProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new RewardedVideoCampaign
            {
                Type = CampaignType.RewardedVideo,
                Name = model.Name,
                Creative = creative,
                Products = products
            });
        }
    }

    internal class LtoRewardedVideoCampaignParser : CampaignParser<RewardedVideoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<RewardProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new RewardedVideoCampaign
            {
                Type = CampaignType.LtoRewardedVideo,
                Name = model.Name,
                Creative = creative,
                Products = products
            });
        }
    }

    internal class BonusCampaignParser : CampaignParser<BonusCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<BonusProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new BonusCampaign
            {
                Type = CampaignType.Bonus,
                Name = model.Name,
                Creative = creative,
                Products = products
            });
        }
    }

    internal class LtoBonusCampaignParser : CampaignParser<BonusCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<BonusProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new BonusCampaign
            {
                Type = CampaignType.LtoBonus,
                Name = model.Name,
                Creative = creative,
                Products = products
            });
        }
    }

    internal class InAppCampaignParser : CampaignParser<InAppCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<InAppProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new InAppCampaign
            {
                Type = CampaignType.InApp,
                Name = model.Name,
                Creative = creative,
                Products = products,
                Store = model.Store,
            });
        }
    }

    internal class LtoInAppCampaignParser : CampaignParser<InAppCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<InAppProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new InAppCampaign
            {
                Type = CampaignType.LtoInApp,
                Name = model.Name,
                Creative = creative,
                Products = products,
                Store = model.Store,
            });
        }
    }

    internal class SubscriptionCampaignParser : CampaignParser<SubscriptionCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<SubscriptionProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new SubscriptionCampaign
            {
                Type = CampaignType.Subscription,
                Name = model.Name,
                Creative = creative,
                Products = products,
            });
        }
    }

    internal class LtoSubscriptionCampaignParser : CampaignParser<SubscriptionCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<SubscriptionProduct>(model, nestedCampaigns, data.ClientId, (creative, products) => new SubscriptionCampaign
            {
                Type = CampaignType.LtoSubscription,
                Name = model.Name,
                Creative = creative,
                Products = products,
            });
        }
    }

    internal class MixedCampaignParser : CampaignParser<MixedCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<ProductDef>(model, nestedCampaigns, data.ClientId, (creative, products) => new MixedCampaign
            {
                Type = CampaignType.Mixed,
                Name = model.Name,
                Creative = creative,
                Products = products,
                Store = model.Store,
            });
        }
    }

    internal class LtoMixedCampaignParser : CampaignParser<MixedCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return Result<ProductDef>(model, nestedCampaigns, data.ClientId, (creative, products) => new MixedCampaign
            {
                Type = CampaignType.LtoMixed,
                Name = model.Name,
                Creative = creative,
                Products = products,
                Store = model.Store,
            });
        }
    }

    internal class ExternalPromoCampaignParser : CampaignParser<ExternalPromoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (model.Info?.PromotedLink == null)
            {
                Logger.LogError($"Invalid model of {nameof(ExternalPromoCampaign)}: {model.Name}. Promoted Link was not found.");
                return false;
            }

            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            var creative = ToCampaignCreative(model.Info!.Creative);
            var products = new List<ExternalLinkProduct>
            {
                new(ProductDef.FakeProductId, model.Info!.PromotedLink!, null, creative.Attributes)
            };
            var parsedProducts = new AbstractProductsCollection(products);
            return new ParsedCampaign(new ExternalPromoCampaign
            {
                Type = CampaignType.External,
                Name = model.Name,
                Creative = creative,
                Products = products,
            }, parsedProducts);
        }
    }

    internal class LtoExternalPromoCampaignParser : CampaignParser<ExternalPromoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (model.Info?.PromotedLink == null)
            {
                Logger.LogError($"Invalid model of {nameof(ExternalPromoCampaign)}: {model.Name}. Promoted Link was not found.");
                return false;
            }

            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            var creative = ToCampaignCreative(model.Info!.Creative);
            var products = new List<ExternalLinkProduct>
            {
                new(ProductDef.FakeProductId, model.Info!.PromotedLink!, null, creative.Attributes)
            };
            var parsedProducts = new AbstractProductsCollection(products);
            return new ParsedCampaign(new ExternalPromoCampaign
            {
                Type = CampaignType.LtoExternal,
                Name = model.Name,
                Creative = creative,
                Products = products,
            }, parsedProducts);
        }
    }

    internal class CrossPromoCampaignParser : CampaignParser<CrossPromoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (string.IsNullOrEmpty(model.Info?.Application?.RawPromoLink))
            {
                Logger.LogError($"Invalid model of {nameof(CrossPromoCampaign)}: {model.Name}. Raw Promo Link was not found.");
                return false;
            }
            if (string.IsNullOrEmpty(model.Info?.Application?.BundleId))
            {
                Logger.LogError($"Invalid model of {nameof(CrossPromoCampaign)}: {model.Name}. BundleId was not found.");
                return false;
            }

            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            var url = model.Info!.Application.GetPromoLink(data.ClientId);
            var creative = ToCampaignCreative(model.Info!.Creative);
            var products = new List<CrossPromoProduct>
            {
                new(ProductDef.FakeProductId, model.Info!.Application!.BundleId!, url, model.Info.Application?.Schemas, null, creative.Attributes)
            };
            var parsedProducts = new AbstractProductsCollection(products);
            return new ParsedCampaign(new CrossPromoCampaign
            {
                Type = CampaignType.CrossPromo,
                Name = model.Name,
                Creative = creative,
                Products = products,
            }, parsedProducts);
        }
    }

    internal class LtoCrossPromoCampaignParser : CampaignParser<CrossPromoCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            if (string.IsNullOrEmpty(model.Info?.Application?.RawPromoLink))
            {
                Logger.LogError($"Invalid model of {nameof(CrossPromoCampaign)}: {model.Name}. Raw Promo Link was not found.");
                return false;
            }
            if (string.IsNullOrEmpty(model.Info?.Application?.BundleId))
            {
                Logger.LogError($"Invalid model of {nameof(CrossPromoCampaign)}: {model.Name}. BundleId was not found.");
                return false;
            }

            return IsValidLimitedOffer(model) && IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            var url = model.Info!.Application.GetPromoLink(data.ClientId);
            var creative = ToCampaignCreative(model.Info!.Creative);
            var products = new List<CrossPromoProduct>
            {
                new(ProductDef.FakeProductId, model.Info!.Application!.BundleId!, url, model.Info.Application?.Schemas, null, creative.Attributes)
            };
            var parsedProducts = new AbstractProductsCollection(products);
            return new ParsedCampaign(new CrossPromoCampaign
            {
                Type = CampaignType.LtoCrossPromo,
                Name = model.Name,
                Creative = creative,
                Products = products,
            }, parsedProducts);
        }
    }

    internal class NotificationCampaignParser : CampaignParser<NotificationCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return IsValidCampaignWithScreen(model);
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return new ParsedCampaign(new NotificationCampaign
            {
                Name = model.Name,
                Creative = ToCampaignCreative(model.Info!.Creative),
            });
        }
    }

    internal class RateReviewCampaignParser : CampaignParser<RateReviewCampaign>
    {
        protected override bool IsValidModel(CampaignModel model)
        {
            return true;
        }

        public override ParsedCampaign Parse(CampaignModel model, IEnumerable<NestedCampaign> nestedCampaigns, AdditionalData data)
        {
            return new ParsedCampaign(new RateReviewCampaign
            {
                Name = model.Name,
            });
        }
    }

    #endregion
}