using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal class ClientIdProvider : IPreInitializable, IInitializable, ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get();

        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly PlatformAPI _platform;
        [CanBeNull]
        private readonly CustomClientId _customClientId;
        [NotNull]
        private readonly UniTaskCompletionSource<bool> _clientIdFromCloudLoadingCompletion = new();
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly ReactiveProperty<bool?> _cloudClientIdReplacedLocal = new();
        [NotNull]
        private readonly ReactiveProperty<string> _commonClientId = new();
        private bool _clientIdFromCloudLoadingResult = false;

        [NotNull]
        public IReadOnlyReactiveProperty<string> ClientId => _generalPrefs.ClientId;

        [CanBeNull]
        public IReadOnlyReactiveProperty<string> CommonClientId => _commonClientId;
        [NotNull]
        public IReadOnlyReactiveProperty<bool> CustomClientIdWasSet => _generalPrefs.CustomClientIdWasSet;
        [NotNull]
        public IReadOnlyReactiveProperty<bool?> CloudClientIdReplacedLocal => _cloudClientIdReplacedLocal;
        [NotNull]
        public IUniTaskSource<bool> ClientIdFromCloudLoadingPromise => _clientIdFromCloudLoadingCompletion;
        public bool SkipClientIdFromCloudLoading { get; set; } = false;
        public uint ClientIdFromCloudLoadingSecondsTimeout { get; set; } = 30;

        public bool IsDisposed => _disposables.IsDisposed;

        public ClientIdProvider(
            [NotNull] GeneralPrefs generalPrefs,
            [NotNull] AppStatePrefs appStatePrefs,
            [NotNull] PlatformAPI platform,
            [CanBeNull] CustomClientId customClientId)
        {
            _generalPrefs = generalPrefs;
            _appStatePrefs = appStatePrefs;
            _platform = platform;
            _customClientId = customClientId;
        }

        void IPreInitializable.PreInitialize()
        {
            if (IsDisposed)
            {
                return;
            }
            TrySetCustomClientId();
        }

        void IInitializable.Initialize()
        {
            if (IsDisposed)
            {
                return;
            }

            if (TrySetCustomClientId())
            {
                return;
            }

            if (_generalPrefs.AuthorizationToken.HasValue
             && !string.IsNullOrEmpty(_generalPrefs.AuthorizationToken.Value))
                _commonClientId.Value = AuthTokenDecoder.RetrieveCommonClientFromToken(_generalPrefs.AuthorizationToken.Value);

            _generalPrefs.AuthorizationToken
                .SkipLatestValueOnSubscribe()
                .Where(token => _generalPrefs.SyncStateEnabled.Value && !string.IsNullOrWhiteSpace(token))
                .Subscribe(AuthTokenChangedHandler)
                .AddTo(_disposables);

            SyncIdWithNativeStorage();
            _generalPrefs.ClientId
                .Subscribe(clientId => _platform.SetValueInNativeStorage(clientId, GeneralPrefs.KeyClientId))
                .AddTo(_disposables);

            TryLoadClientIdFromCloudAsync(_disposables.GetOrCreateToken()).Forget();
        }

        private bool TrySetCustomClientId()
        {
            if (_customClientId != null)
            {
                var customClientId = _customClientId.ToString();
                if (ClientId.HasValue)
                {
                    if (customClientId != ClientId.Value)
                        throw new MagifyClientIdCannotBeChangedException();
                    return true;
                }
                SetClientId(_customClientId.ToString());
                _generalPrefs.CustomClientIdWasSet.Value = true;
                return true;
            }
            return false;
        }

        void IDisposable.Dispose()
        {
            if (IsDisposed)
            {
                return;
            }
            _disposables.Release();
        }

        private void SetClientId([NotNull] string value)
        {
            try
            {
                _generalPrefs.ClientId.Value = value;
            }
            catch (Exception e)
            {
                _logger.LogError($"Exception caught during client id setting: {e.Message}");
                _logger.LogException(e);
            }
        }

        private void SyncIdWithNativeStorage()
        {
            if (string.IsNullOrEmpty(_generalPrefs.ClientId.Value))
            {
                var nativeStorageValue = _platform.GetValueInNativeStorage(GeneralPrefs.KeyClientId);
                if (string.IsNullOrEmpty(nativeStorageValue))
                {
                    SetClientId(nativeStorageValue = Guid.NewGuid().ToString());
                    _platform.SetValueInNativeStorage(nativeStorageValue, GeneralPrefs.KeyClientId);
                }
                else
                {
                    _logger.Log($"Client id is retrieved from native storage: {nativeStorageValue}");
                    _generalPrefs.IsClientIdRetrievedFromNativeStorage = true;
                    SetClientId(nativeStorageValue);
                }
            }
        }

        private async UniTask TryLoadClientIdFromCloudAsync(CancellationToken cancellationToken)
        {
            _logger.Log($"{nameof(TryLoadClientIdFromCloudAsync)} has been called");
            if (SkipClientIdFromCloudLoading)
            {
                _logger.Log($"Skipping clientId from cloud loading.");
                _clientIdFromCloudLoadingCompletion.TrySetResult(false);
                return;
            }
            if (_platform.RuntimePlatform is not RuntimePlatform.IPhonePlayer)
            {
                // Now we have implementation only for iOS
                _logger.Log($"Platform {_platform.RuntimePlatform} is not supported for loading of client if from cloud, skipping.");
                _clientIdFromCloudLoadingCompletion.TrySetResult(false);
                return;
            }
            if (!_generalPrefs.SyncStateEnabled.Value)
            {
                _logger.Log($"Can't load client id. {SyncStateResult.SyncIsDisabled().ErrorMessage}");
                _clientIdFromCloudLoadingCompletion.TrySetResult(false);
                return;
            }
            if (_appStatePrefs.CloudClientIdFetched.Value)
            {
                _logger.Log("Client id has successfully been fetched in the past, skipping trying to fetch it again.");
                _clientIdFromCloudLoadingCompletion.TrySetResult(false);
                return;
            }

            _logger.Log($"Trying to load client id from cloud with timeout {ClientIdFromCloudLoadingSecondsTimeout}...");
            var isLoaded = false;
            var replacedLocalId = default(bool?);
            try
            {
                var firstLaunchTracked = _generalPrefs.IsFirstAppLaunchTracked.Value;
                while (firstLaunchTracked is not true)
                {
                    firstLaunchTracked = await _generalPrefs.IsFirstAppLaunchTracked;
                }
                _logger.Log("Request client id from cloud...");
                var clientId = await _platform.GetValueFromCloudStorage(CloudStorageKeys.ClientId, ClientIdFromCloudLoadingSecondsTimeout, cancellationToken);
                _logger.Log($"Requested client id from cloud result: {clientId}");
                await TaskScheduler.SwitchToMainThread(cancellationToken);
                if (string.IsNullOrEmpty(clientId))
                {
                    _platform.SetValueInCloudStorage(CloudStorageKeys.ClientId, _generalPrefs.ClientId.Value);
                    replacedLocalId = false;
                }
                else
                {
                    isLoaded = true;
                    replacedLocalId = _cloudClientIdReplacedLocal.Value = _generalPrefs.ClientId.Value != clientId;
                    _appStatePrefs.IsFbAuthorizationCompleted.Value = false;
                    SetClientId(clientId);
                }
                if (_appStatePrefs.IsAutoRestoreStateEnabled.Value)
                {
                    // We want auto-restore if ClientId WAS Retrieved from native storage (keychain) or iCloud
                    _appStatePrefs.IsPendingRestore.Value = isLoaded || _generalPrefs.IsClientIdRetrievedFromNativeStorage;
                }
                _appStatePrefs.CloudClientIdFetched.Value = true;
            }
            catch (OperationCanceledException e)
            {
                _logger.LogWarning("Loading of the client id from cloud has been cancelled.");
                _clientIdFromCloudLoadingCompletion.TrySetCanceled(e.CancellationToken);
            }
            catch (Exception e)
            {
                _logger.LogException(new FailedToLoadClientIdFromCloudException(e));
            }
            finally
            {
                _logger.Log("Loading client id from cloud finished");
                _cloudClientIdReplacedLocal.Value = replacedLocalId;
                _clientIdFromCloudLoadingResult = isLoaded;
                _clientIdFromCloudLoadingCompletion.TrySetResult(isLoaded);
            }
        }

        /// <summary>
        /// Will be finished in both: if loaded or loading failed
        /// </summary>
        /// <returns>
        /// True - if loaded
        /// False - loading failed
        /// </returns>
        public async UniTask<bool> WaitForLoadingFromCloudFinished(CancellationToken cancellationToken)
        {
            return await _clientIdFromCloudLoadingCompletion.Task.AttachExternalCancellation(cancellationToken);
        }

        private void AuthTokenChangedHandler([NotNull] string authToken)
        {
            try
            {
                var clientId = AuthTokenDecoder.RetrieveClientFromToken(authToken);
                if (clientId == null || clientId == ClientId.Value)
                {
                    _logger.Log("Client id won't be updated from auth token.");
                }
                else
                {
                    // Don't do it async, it's important to immediately update data
                    if (_appStatePrefs.IsAutoRestoreStateEnabled.Value)
                    {
                        _appStatePrefs.IsPendingRestore.Value = true;
                    }
                    _appStatePrefs.IsFbAuthorizationCompleted.Value = false;
                    SetClientId(clientId);
                }
            }
            catch (TooShortMagifyTokenException e)
            {
                _logger.LogError("Failed to retrieve client id from auth token. See details below.");
                _logger.LogException(e);
            }
            catch (DecodeException e)
            {
                _logger.LogError("Failed to retrieve client id from auth token. See details below.");
                _logger.LogException(e);
            }

            try
            {
                var commonClientId = AuthTokenDecoder.RetrieveCommonClientFromToken(authToken);
                _commonClientId.Value = commonClientId;
            }
            catch (TooShortMagifyTokenException e)
            {
                _logger.LogError("Failed to retrieve common client id from auth token. See details below.");
                _logger.LogException(e);
            }
            catch (DecodeException e)
            {
                _logger.LogError("Failed to retrieve common client id from auth token. See details below.");
                _logger.LogException(e);
            }
        }
    }
}