using System;
using System.Collections.Generic;
using System.IO;
using System.Text;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Model;
using Magify.Rx;
using Newtonsoft.Json;
using UnityEngine;

namespace Magify
{
    internal class AppStateManager : IAppStateManager, IInitializable, IDisposable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(LoggingScope.AppState);

        [NotNull]
        private readonly PlatformAPI _platform;
        [NotNull]
        private readonly IServerApi _serverApi;
        [NotNull]
        private readonly GeneralPrefs _generalPrefs;
        [NotNull]
        private readonly ClientIdProvider _clientIdProvider;
        [NotNull]
        private readonly CountersStorage _counters;
        [NotNull]
        private readonly AppStatePrefs _appStatePrefs;
        [NotNull]
        private readonly ClientStateConfig _clientStateConfig;
        [NotNull]
        private readonly AppStateBuilder _appStateBuilder;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly Subject<SyncStateResult> _restoreStateCompletedSubject = new();

        [CanBeNull]
        private CancellationTokenSource _saveStateCts;
        [CanBeNull]
        private CancellationTokenSource _restoreStateCts;
        [CanBeNull]
        private CancellationTokenSource _initialRestoreCts;
        private bool _restoreStateCompletedCallbackInProgress;
        private bool _isInitialRestoreInProgress;

        [NotNull]
        public IObservable<SyncStateResult> OnRestoreStateCompleted => _restoreStateCompletedSubject;
        [NotNull]
        public IReactiveProperty<bool> SyncStateEnabled => _generalPrefs.SyncStateEnabled;
        [NotNull]
        public IReactiveProperty<AutoRestoreStateInfo> AutoRestoreStateInfo { get; } =
            new ReactiveProperty<AutoRestoreStateInfo>(Magify.AutoRestoreStateInfo.NotStarted());
        [NotNull]
        public IReactiveProperty<bool> IsAutoRestoreStateEnabled => _appStatePrefs.IsAutoRestoreStateEnabled;
        public bool HasSocialAuthorizationData => !string.IsNullOrEmpty(_appStatePrefs.ExternalAuthToken.Value)
                                               && !string.IsNullOrEmpty(_appStatePrefs.ExternalAuthProvider.Value);

        public AppStateManager(
            [NotNull] PlatformAPI platform,
            [NotNull] IServerApi serverApi,
            [NotNull] GeneralPrefs generalPrefs,
            [NotNull] ClientIdProvider clientIdProvider,
            [NotNull] CountersStorage counters,
            [NotNull] AppStatePrefs appStatePrefs,
            [NotNull] ClientStateConfig clientStateConfig,
            [NotNull] string storagePath)
        {
            _platform = platform;
            _serverApi = serverApi;
            _generalPrefs = generalPrefs;
            _clientIdProvider = clientIdProvider;
            _counters = counters;
            _appStatePrefs = appStatePrefs;
            _clientStateConfig = clientStateConfig;
            _appStateBuilder = new AppStateBuilder(generalPrefs, counters, clientStateConfig, storagePath);
            SyncStateEnabled
                .Subscribe(enabled => _logger.Log($"Sync state {(enabled ? "enabled" : "disabled")}"))
                .AddTo(_disposables);
            IsAutoRestoreStateEnabled
                .Subscribe(enabled => _logger.Log($"Auto restore state is {(enabled ? "enabled" : "disabled")}"))
                .AddTo(_disposables);
            IsAutoRestoreStateEnabled
                .Where(enabled => enabled is false)
                .Subscribe(_ => _initialRestoreCts?.Cancel())
                .AddTo(_disposables);
        }

        async void IInitializable.Initialize()
        {
            _clientIdProvider.ClientId
                .Where(token => SyncStateEnabled.Value && !string.IsNullOrWhiteSpace(token))
                .Subscribe(ClientIdChangedHandler)
                .AddTo(_disposables);

            await _clientIdProvider.WaitForLoadingFromCloudFinished(_disposables.GetOrCreateToken());
            if (string.IsNullOrEmpty(_generalPrefs.AuthorizationToken.Value))
            {
                UniTask.RunOnThreadPool(() => _serverApi.GetAuthorizationTokenUntilSuccessAsync(_disposables.GetOrCreateToken())).Forget();
            }
            if (_appStatePrefs.IsPendingRestore.Value)
            {
                LaunchInitialStateRestore(_disposables.GetOrCreateToken()).Forget();
            }
        }

        void IDisposable.Dispose()
        {
            _disposables.Release();
            _saveStateCts?.Cancel();
            _saveStateCts?.Dispose();
            _restoreStateCts?.Cancel();
            _restoreStateCts?.Dispose();
        }

        public async UniTask<bool> RequestSocialAuthTokenFor([NotNull] string provider, [NotNull] string token, CancellationToken cancellationToken)
        {
            if (!SyncStateEnabled.Value)
            {
                _logger.LogError($"Social authorization failed: {SyncStateResult.SyncIsDisabled()}");
                return false;
            }
            _appStatePrefs.UpdateExternalAuthToken(provider, token);
            _serverApi.ResetAuthorizationToken();
            try
            {
                var authToken = await _serverApi.GetAuthorizationTokenUntilSuccessAsync(LinkCancellationToken(cancellationToken).Token);
                await TaskScheduler.SwitchToMainThread(cancellationToken);
                if (authToken != null) _logger.Log($"Authorization token updated with external user token. Provider: {provider}; Token: {token}; Auth token: {authToken}");
                else _logger.LogError($"Failed to update authorization token with external user token. Provider: {provider}; Token: {token}");
                return authToken != null;
            }
            catch (MagifyAuthTokenLoadingCancelledException e)
            {
                _logger.LogException(e);
                return false;
            }
        }

        public void ResetSocialAuthToken()
        {
            _appStatePrefs.ResetExternalAuthToken();
            _serverApi.ResetAuthorizationToken();
        }

        public async UniTask<SyncStateResult> SaveState(int? weight, CancellationToken cancellationToken)
        {
            _logger.Log($"Start saving application state. Weight: {weight?.ToString() ?? "null"}");
            try
            {
                await _clientIdProvider.WaitForLoadingFromCloudFinished(cancellationToken);
                if (cancellationToken.IsCancellationRequested)
                {
                    return SyncStateResult.SaveCancelled(cancellationToken, weight);
                }
            }
            catch (OperationCanceledException e)
            {
                return SyncStateResult.SaveCancelled(e.CancellationToken, weight);
            }

            SyncStateResult result;
            if (!SyncStateEnabled.Value)
            {
                result = SyncStateResult.SyncIsDisabled();
                _logger.LogError($"Can't save state. {result.ErrorMessage}");
                return result;
            }
            if (_appStatePrefs.IsPendingRestore.Value)
            {
                result = SyncStateResult.RestoreInProgress();
                _logger.LogError($"Can't save state. {result.ErrorMessage}");
                return result;
            }

            try
            {
                cancellationToken = LinkCancellationToken(cancellationToken).Token;
                var authResult = await AuthorizeUserIfNeeded(cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();
                if (authResult.Kind is SyncStateResult.ResultKind.Cancelled)
                {
                    return SyncStateResult.SaveCancelled(authResult.CancellationToken ?? CancellationToken.None, weight);
                }

                if (authResult.IsSuccess)
                {
                    ThreadingUtils.CancelAndReplace(ref _saveStateCts, CancellationTokenSource.CreateLinkedTokenSource(cancellationToken));
                    cancellationToken = _saveStateCts!.Token;
                    var response = await _serverApi.SaveState(await _appStateBuilder.Build(cancellationToken, _platform.RuntimePlatform), weight, cancellationToken);
                    switch (response.Status)
                    {
                        case SaveStateRequestResponse.StatusCode.Success:
                            _logger.Log($"Application state successfully saved. Weight: {weight?.ToString() ?? "null"}");
                            return SyncStateResult.Ok(weight, internalStateRestored: false);
                        case SaveStateRequestResponse.StatusCode.Conflict:
                            result = SyncStateResult.SaveConflict(weight);
                            _logger.Log(result.ErrorMessage);
                            return result;
                        case SaveStateRequestResponse.StatusCode.ForceUpdateRequest:
                            result = SyncStateResult.ForceUpdateRequested(weight);
                            _logger.Log(result.ErrorMessage);
                            return result;
                        case SaveStateRequestResponse.StatusCode.Fail:
                            break;
                    }
                }

                result = SyncStateResult.FailedToSave($"weight: {weight?.ToString() ?? "null"}", weight);
                _logger.LogError(result.ErrorMessage);
                return result;
            }
            catch (OperationCanceledException e)
            {
                return SyncStateResult.SaveCancelled(e.CancellationToken, weight);
            }
        }

        public async UniTask<SyncStateResult> RestoreState(int? weight, CancellationToken cancellationToken, bool forceSync = false)
        {
            _logger.Log($"Start restoring application state. Weight: {weight?.ToString() ?? "null"}");
            try
            {
                await _clientIdProvider.WaitForLoadingFromCloudFinished(cancellationToken);
                if (cancellationToken.IsCancellationRequested)
                {
                    return SyncStateResult.RestoreCancelled(cancellationToken, weight);
                }
            }
            catch (OperationCanceledException e)
            {
                return SyncStateResult.RestoreCancelled(e.CancellationToken, weight);
            }

            SyncStateResult result;
            if (_restoreStateCompletedCallbackInProgress)
            {
                result = SyncStateResult.RestoreCallbackInProgress(weight);
                _logger.LogError($"Can't restore state. {result.ErrorMessage}");
                return result;
            }
            if (!SyncStateEnabled.Value && !forceSync)
            {
                result = SyncStateResult.SyncIsDisabled();
                _logger.LogError(result.ErrorMessage);
                return result;
            }
            if (!forceSync && _appStatePrefs.IsPendingRestore.Value && weight != null)
            {
                result = SyncStateResult.RestoreInProgress();
                _logger.LogError($"Can't restore state. {result.ErrorMessage}");
                return result;
            }

            try
            {
                _appStatePrefs.IsRestoreInProgress.Value = true;
                cancellationToken = LinkCancellationToken(cancellationToken).Token;
                var authResult = await AuthorizeUserIfNeeded(cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();
                if (authResult.Kind is SyncStateResult.ResultKind.Cancelled)
                {
                    return SyncStateResult.RestoreCancelled(authResult.CancellationToken ?? CancellationToken.None, weight);
                }

                if (authResult.IsSuccess)
                {
                    ThreadingUtils.CancelAndReplace(ref _restoreStateCts, CancellationTokenSource.CreateLinkedTokenSource(cancellationToken));
                    cancellationToken = _restoreStateCts!.Token;
                    var applicationState = await _serverApi.RestoreState(weight, cancellationToken);
                    var internalStateRestored = false;
                    if (applicationState.Status is RestoreStateRequestResponse.StatusCode.Success && applicationState.Result != null)
                    {
                        _logger.Log($"Application state successfully loaded. Weight: {weight?.ToString() ?? "null"}");
                        var parsedAppState = _appStateBuilder.Parse(applicationState.Result, _platform.RuntimePlatform);
                        using (new DebugAppStateRestoreScope())
                        {
                            internalStateRestored = await ApplyApplicationState(parsedAppState, applicationState.Weight, cancellationToken);
                            cancellationToken.ThrowIfCancellationRequested();
                        }
                        _logger.Log($"Application state successfully restored. Weight: {weight?.ToString() ?? "null"}");
                    }
                    else
                    {
                        _logger.Log("Application state isn't successfully loaded, skipping writing to files.");
                    }
                    _appStatePrefs.IsPendingRestore.Value = false;
                    _appStatePrefs.IsRestoreInProgress.Value = false;
                    _restoreStateCompletedCallbackInProgress = true;
                    result = applicationState.Status switch
                    {
                        RestoreStateRequestResponse.StatusCode.Success when applicationState.Result is null => SyncStateResult.NothingToRestore(weight),
                        RestoreStateRequestResponse.StatusCode.Success => SyncStateResult.Ok(weight, internalStateRestored),
                        RestoreStateRequestResponse.StatusCode.AlreadyHasNewest => SyncStateResult.AlreadyHasNewestProgress(weight),
                        RestoreStateRequestResponse.StatusCode.OnlyForceRestorePossible => SyncStateResult.OnlyForceRestorePossible(weight),
                        _ => SyncStateResult.FailedToRestore("unexprected error", weight),
                    };
                    result.RestoredWeight = applicationState.Weight;
                    _restoreStateCompletedSubject.OnNext(result);
                    _restoreStateCompletedCallbackInProgress = false;
                    return result;
                }

                result = SyncStateResult.FailedToRestore($"weight: {weight?.ToString() ?? "null"} , content: null", weight);
                _logger.LogError(result.ErrorMessage);
                return result;
            }
            catch (OperationCanceledException e)
            {
                return SyncStateResult.RestoreCancelled(e.CancellationToken, weight);
            }
            catch (MagifyFailedToParseRestoreProgressResponseException e)
            {
                return SyncStateResult.FailedToRestore(e.Message, weight);
            }
            finally
            {
                _restoreStateCompletedCallbackInProgress = false;
                _appStatePrefs.IsRestoreInProgress.Value = false;
            }
        }

        private async UniTask<bool> ApplyApplicationState([CanBeNull] ApplicationState state, [CanBeNull] int? weight, CancellationToken cancellationToken)
        {
            _logger.Log($"Applying application state: \n{JsonFacade.SerializeObject(state, Formatting.Indented)}");
            if (state?.ClientState != null && _clientStateConfig.AppStateProvider != null)
            {
                try
                {
                    await _clientStateConfig.AppStateProvider.HandleState(state.ClientState, weight, cancellationToken);
                    await TaskScheduler.SwitchToMainThread(cancellationToken);
                }
                catch (Exception e)
                {
                    var exception = new ProviderFailedToHandleClientStateException(_clientStateConfig.AppStateProvider, e);
                    _logger.LogException(exception);
                    throw exception;
                }
            }
            var sdk = state?.Sdk(_platform.RuntimePlatform);
            var internalStateRestored = sdk is { IsEmpty: false };
            if (sdk?.GeneralPrefs != null)
            {
                _logger.Log("Applying general prefs");
                _generalPrefs.RewriteContent(sdk.GeneralPrefs);
            }
            if (sdk?.CountersPrefs != null)
            {
                _logger.Log("Applying counters prefs");
                _counters.RewriteContent(sdk.CountersPrefs);
            }
            return internalStateRestored;
        }

        [ItemNotNull]
        private async UniTask<SyncStateResult> AuthorizeUserIfNeeded(CancellationToken cancellationToken)
        {
            if (_appStatePrefs.IsFbAuthorizationCompleted.Value)
            {
                return SyncStateResult.Ok(weight: null, internalStateRestored: false);
            }
            if (cancellationToken.IsCancellationRequested)
            {
                return SyncStateResult.AuthorizationCancelled(cancellationToken);
            }

            try
            {
                _logger.Log($"Internal client authorization is required: {_clientIdProvider.ClientId.Value}");

                var userId = await _serverApi.AuthorizeUser(_clientIdProvider.ClientId.Value, cancellationToken);
                cancellationToken.ThrowIfCancellationRequested();

                if (userId == null)
                {
                    var result = SyncStateResult.FailedToAuthorize(_clientIdProvider.ClientId.Value);
                    _logger.LogError(result.ErrorMessage);
                    return result;
                }
                _appStatePrefs.IsFbAuthorizationCompleted.Value = true;
                _logger.Log($"Authorization successfully completed. User id: {userId}");
                return SyncStateResult.Ok(weight: null, internalStateRestored: false);
            }
            catch (OperationCanceledException e)
            {
                return SyncStateResult.AuthorizationCancelled(e.CancellationToken);
            }
        }

        private void ClientIdChangedHandler([NotNull] string authToken)
        {
            if (_appStatePrefs.IsPendingRestore.Value)
            {
                LaunchInitialStateRestore(_disposables.GetOrCreateToken()).Forget();
            }
        }

        private async UniTaskVoid LaunchInitialStateRestore(CancellationToken cancellationToken)
        {
            _logger.Log($"{nameof(LaunchInitialStateRestore)} called");
            if (IsAutoRestoreStateEnabled.Value is false)
            {
                _logger.Log("Auto restore state is disabled.");
                _appStatePrefs.IsPendingRestore.Value = false;
                return;
            }
            if (_isInitialRestoreInProgress)
            {
                _logger.Log("Auto restore state already ran.");
                return;
            }
            try
            {
                _initialRestoreCts?.Cancel();
                _initialRestoreCts?.Dispose();
                _initialRestoreCts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                cancellationToken = _initialRestoreCts.Token;
                AutoRestoreStateInfo.Value = Magify.AutoRestoreStateInfo.InProgress();
                _isInitialRestoreInProgress = true;
                bool continueRestore;
                var backoff = new ExponentialBackoff();
                do
                {
                    var result = await RestoreState(null, cancellationToken, forceSync: true)!;
                    continueRestore = result?.Kind is not (SyncStateResult.ResultKind.Success or SyncStateResult.ResultKind.NothingToRestore);
                    if (continueRestore)
                    {
                        _logger.LogWarning($"Failed to restore initial state. {result?.ErrorMessage}");
                        await UniTask.Delay(Convert.ToInt32(backoff.NextDelay()), DelayType.Realtime, cancellationToken: cancellationToken);
                    }
                    else
                    {
                        AutoRestoreStateInfo.Value = result.Kind switch
                        {
                            SyncStateResult.ResultKind.Success => Magify.AutoRestoreStateInfo.Success(),
                            SyncStateResult.ResultKind.NothingToRestore => Magify.AutoRestoreStateInfo.NothingToRestore(),
                            _ => Magify.AutoRestoreStateInfo.Failed("Unknown restore result kind", default),
                        };
                    }

                } while (continueRestore);
            }
            catch (OperationCanceledException)
            {
                AutoRestoreStateInfo.Value = Magify.AutoRestoreStateInfo.Cancelled();
            }
            catch (Exception e)
            {
                AutoRestoreStateInfo.Value = Magify.AutoRestoreStateInfo.Failed(e.Message, e);
            }
            finally
            {
                _isInitialRestoreInProgress = false;
            }
        }

        [NotNull]
        private CancellationTokenSource LinkCancellationToken(CancellationToken cancellationToken)
        {
            return CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, _disposables.GetOrCreateToken());
        }

        [NotNull]
        private static string ErrIfCancelled(CancellationToken cancellationToken)
        {
            return cancellationToken.IsCancellationRequested ? "(request cancelled)" : string.Empty;
        }

        private class DebugAppStateRestoreScope : IDisposable
        {
#if MAGIFY_VERBOSE_LOGGING
            [CanBeNull]
            private readonly IDisposable _disposable;
            [NotNull]
            private readonly Dictionary<string, string> _map = new();
#endif

#if MAGIFY_VERBOSE_LOGGING
            public DebugAppStateRestoreScope()
            {
                _disposable = BinaryStorage.OnStorageSavedAsJson
                    .Subscribe(tuple => _map.TryAdd(tuple.Name, tuple.Json));
            }
#endif

            public void Dispose()
            {
#if MAGIFY_VERBOSE_LOGGING
                _disposable?.Dispose();
                var builder = new StringBuilder();
                builder.Append("Prettified restored progress:");
                foreach (var (key, value) in _map)
                {
                    builder
                        .Append('\n')
                        .Append(key)
                        .Append(':')
                        .Append('\n')
                        .Append(value)
                        .Append('\n')
                        .Append("--------------------");
                }
                var str = builder.ToString();
                _logger.Log(str);
                var path = Path.Combine(PackageInfo.PersistentPath, "pretty-progress.txt");
                if (!Directory.Exists(Path.GetDirectoryName(path)))
                    Directory.CreateDirectory(Path.GetDirectoryName(path));
                File.WriteAllText(path, str);
#endif
            }
        }
/*
        private class RestoreProgressScopeSource
        {
            [NotNull]
            private readonly AppStatePrefs _appStatePrefs;
            [NotNull]
            private readonly object _lock = new();
            private int _counter;

            public RestoreProgressScopeSource([NotNull] AppStatePrefs appStatePrefs)
            {
                _appStatePrefs = appStatePrefs;
            }

            [NotNull]
            public Token Get()
            {
                lock (_lock)
                {
                    if (_counter == 0)
                    {
                        _appStatePrefs.IsUserRestoreInProgress.Value = true;
                    }
                    var token = new Token(OnTokenDisposed);
                    _counter++;
                    return token;
                }
            }

            private void OnTokenDisposed()
            {
                lock (_lock)
                {
                    _counter--;
                    if (_counter == 0)
                    {
                        _appStatePrefs.IsUserRestoreInProgress.Value = false;
                    }
                }
            }

            public class Token : IDisposable
            {
                [NotNull]
                private readonly object _lock = new();
                private readonly Action _onCurrentDisposed;
                private bool _disposed;

                public Token([NotNull] Action onCurrentDisposed)
                {
                    _onCurrentDisposed = onCurrentDisposed;
                }

                public void Dispose()
                {
                    lock (_lock)
                    {
                        if (!_disposed)
                        {
                            _disposed = true;
                            _onCurrentDisposed?.Invoke();
                        }
                    }
                }
            }
        }
        */
    }
}