﻿using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class DownloadPromise<T> : ICancelable
    {
        internal class Handle : IDisposable
        {
            [NotNull]
            private readonly Action _onDisposed;
            [NotNull]
            private readonly CancellationTokenSource _cts = new();
            [NotNull]
            private readonly UniTaskCompletionSource<T> _completionSource = new();

            public UniTask<T> Task => _completionSource.Task;

            public Handle([NotNull] Action onDisposed, UniTask<T> promiseTask)
            {
                _onDisposed = onDisposed;
                Wait(promiseTask).Forget();
            }

            private async UniTaskVoid Wait(UniTask<T> original)
            {
                try
                {
                    _completionSource.TrySetResult(await original);
                }
                catch (OperationCanceledException e)
                {
                    _completionSource.TrySetCanceled(e.CancellationToken);
                }
                catch (Exception e)
                {
                    _completionSource.TrySetException(e);
                }
            }

            public void Dispose()
            {
                _cts.Cancel();
                _completionSource.TrySetCanceled(_cts.Token);
                _cts.Dispose();
                _onDisposed();
            }
        }

        [NotNull]
        private readonly UniTaskCompletionSource<T> _promise = new();
        [NotNull]
        private readonly CancellationTokenSource _cancellationTokenSource = new();
        [NotNull]
        private readonly object _gate = new();
        private int _handles;

        // ReSharper disable once InconsistentNaming
        public CancellationToken OnDisposedCancellationToken => _cancellationTokenSource.Token;
        public bool IsCancellationRequested => _cancellationTokenSource.IsCancellationRequested;
        public bool IsDisposed { get; private set; }

        public bool TrySetResult(T result)
        {
            return _promise.TrySetResult(result);
        }

        public Handle GetHandle()
        {
            lock (_gate)
            {
                _handles++;
                return new Handle(HandleDisposed, _promise.Task);
            }
        }

        private void HandleDisposed()
        {
            lock (_gate)
            {
                _handles--;
            }
            if (_handles == 0)
            {
                Dispose();
            }
        }

        public void Dispose()
        {
            if (IsDisposed)
                return;
            IsDisposed = true;
            _promise.TrySetCanceled();
            _cancellationTokenSource.Cancel();
            _cancellationTokenSource.Dispose();
        }
    }
}