﻿using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify
{
    public partial class MagifyManager
    {
        public static class Authorization
        {
            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Promise to return the result of an attempt load ClientId from native cloud storage.
            /// </summary>
            [NotNull]
            public static IUniTaskSource<bool> ClientIdFromCloudLoadingPromise
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(ClientIdFromCloudLoadingPromise));
                    return MagifyPlatformAPI!.ClientIdFromCloudLoadingPromise;
                }
            }

            /// <summary>
            /// Check whether social provider and token have been passed via <see cref="AuthorizeUser"/>.
            /// </summary>
            /// <remarks>
            /// Will became `false` again after calling <see cref="ResetUserAuthorization"/>.
            /// </remarks>
            public static bool HasSocialAuthorizationData
            {
                get
                {
                    ThrowIfMagifyIsNotReady(nameof(HasSocialAuthorizationData));
                    return MagifyPlatformAPI!.HasSocialAuthorizationData;
                }
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Requests authorization with the social authorization data.
            /// </summary>
            /// <param name="provider">The name of the social authorization method. You can select it yourself depends on your environment.</param>
            /// <param name="token">A unique token for each user that will be associated with the current user account.</param>
            /// <param name="cancellationToken">Pass cancellation token if you want to cancel this operation, for example by timeout.</param>
            /// <returns>
            /// Result of the authorization operation. True if successful.
            /// </returns>
            /// <example>
            /// <code>
            /// AuthorizeUser(provider: "google", token: "3A2u6a6fZoxHP7n0AosZbv3k8zvy0Y", cancellationToken);
            /// </code>
            /// </example>
            public static UniTask<bool> AuthorizeUser([NotNull] string provider, [NotNull] string token, CancellationToken cancellationToken)
            {
                ThrowIfMagifyIsNotReady(nameof(AuthorizeUser));
                if (string.IsNullOrEmpty(provider) || string.IsNullOrEmpty(token))
                {
                    _logger!.LogWarning($"Parameters {nameof(provider)} and {nameof(token)} cannot be null or empty, skipping");
                    return UniTask.FromResult(false);
                }
                _logger!.Log($"Authorize user with {nameof(provider)}={provider}; {nameof(token)}={token}");
                return MagifyPlatformAPI!.RequestSocialAuthTokenFor(provider, token, cancellationToken);
            }

            /// <inheritdoc cref="ThrowIfMagifyIsNotReady"/>
            /// <summary>
            /// Resets the authorization data (provider and token) that were set earlier via <see cref="AuthorizeUser"/>.
            /// From now, saving and restoring progress will work on the basis of `ClientId` only, until next <see cref="AuthorizeUser"/> call.
            /// </summary>
            public static void ResetUserAuthorization()
            {
                ThrowIfMagifyIsNotReady(nameof(ResetUserAuthorization));
                _logger!.Log("Reset user authorization");
                MagifyPlatformAPI!.ResetSocialAuthToken();
            }
        }
    }
}