﻿using System;
using JetBrains.Annotations;

namespace Magify.Rx
{
    internal class SkipObservable<T> : OperatorObservableBase<T>
    {
        [NotNull]
        private readonly IObservable<T> _source;
        private readonly int _count;

        public SkipObservable([NotNull] IObservable<T> source, int count)
        {
            _source = source;
            _count = count;
        }

        // optimize combiner
        [NotNull]
        public IObservable<T> Combine(int count)
        {
            // use sum
            // xs = 6
            // xs.Skip(2) = 4
            // xs.Skip(2).Skip(3) = 1
            return new SkipObservable<T>(_source, _count + count);
        }

        [NotNull]
        protected override IDisposable SubscribeCore([NotNull] IObserver<T> observer, [NotNull] IDisposable cancel)
        {
            return _source.Subscribe(new Skip(this, observer, cancel));
        }

        private class Skip : OperatorObserverBase<T, T>
        {
            private int _remaining;

            public Skip([NotNull] SkipObservable<T> parent, [NotNull] IObserver<T> observer, [NotNull] IDisposable cancel) : base(observer, cancel)
            {
                _remaining = parent._count;
            }

            public override void OnNext(T value)
            {
                if (_remaining <= 0)
                {
                    Observer.OnNext(value);
                }
                else
                {
                    _remaining--;
                }
            }

            public override void OnError(Exception error)
            {
                try { Observer.OnError(error); } finally { Dispose(); }
            }

            public override void OnCompleted()
            {
                try { Observer.OnCompleted(); } finally { Dispose(); }
            }
        }
    }
}