﻿using System;
using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;

namespace Magify
{
    internal static class ServerTimeProviderExtensions
    {
        public static async UniTask<DateTime?> TryGetServerTime(this IReadOnlyCollection<IServerTimeProvider> providers, CancellationToken cancellationToken)
        {
            var internalCancellationToken = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            var completionSource = new UniTaskCompletionSource<DateTime?>();
            var failedRequestsNumber = 0;
            var providersNumber = providers.Count;
            foreach (var provider in providers)
            {
                Request(provider, internalCancellationToken.Token, completionSource, onRequestFailed).Forget();
            }

            var result = await completionSource.Task;
            cancel();
            return result;

            void onRequestFailed()
            {
                failedRequestsNumber++;
                if (failedRequestsNumber >= providersNumber)
                {
                    completionSource.TrySetResult(null);
                    cancel();
                }
            }

            void cancel()
            {
                internalCancellationToken.Cancel();
                internalCancellationToken.Dispose();
            }
        }

        private static async UniTaskVoid Request(
            IServerTimeProvider provider,
            CancellationToken cancellationToken,
            UniTaskCompletionSource<DateTime?> successCompletionSource,
            Action failCallback)
        {
            var result = await provider.TryGetServerTime(cancellationToken);
            if (result.HasValue)
            {
                successCompletionSource.TrySetResult(result.Value);
            }
            else
            {
                failCallback?.Invoke();
            }
        }
    }
}