using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    internal class LimitedTimeOfferNativeEventsAccumulator
    {
        private readonly Queue<(EventKind Kind, LtoInfo Offer)> _eventsQueue = new();
        private readonly CompositeDisposable _disposables = new();

        public LimitedTimeOfferNativeEventsAccumulator()
        {
            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnAdded += handler, handler => MagifyManager.Lto.OnAdded -= handler)
                .Subscribe(item => _eventsQueue.Enqueue((EventKind.Added, item)))
                .AddTo(_disposables);
            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnRemoved += handler, handler => MagifyManager.Lto.OnRemoved -= handler)
                .Subscribe(item => _eventsQueue.Enqueue((EventKind.Removed, item)))
                .AddTo(_disposables);
            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnUpdated += handler, handler => MagifyManager.Lto.OnUpdated -= handler)
                .Subscribe(item => _eventsQueue.Enqueue((EventKind.Updated, item)))
                .AddTo(_disposables);
            Observable
                .FromEvent<LtoInfo>(handler => MagifyManager.Lto.OnFinished += handler, handler => MagifyManager.Lto.OnFinished -= handler)
                .Subscribe(item => _eventsQueue.Enqueue((EventKind.Finished, item)))
                .AddTo(_disposables);
        }

        public void StopEventsCatching()
        {
            _disposables?.Dispose();
        }

        public void SpendCurrentEvents(
            [NotNull] Action<LtoInfo> addedHandler,
            [NotNull] Action<LtoInfo> removedHandler,
            [NotNull] Action<LtoInfo> updatedHandler,
            [NotNull] Action<LtoInfo> finishedHandler)
        {
            while (_eventsQueue.TryDequeue(out var item))
            {
                switch (item.Kind)
                {
                    case EventKind.Added:
                        addedHandler(item.Offer);
                        break;
                    case EventKind.Removed:
                        removedHandler(item.Offer);
                        break;
                    case EventKind.Updated:
                        updatedHandler(item.Offer);
                        break;
                    case EventKind.Finished:
                        finishedHandler(item.Offer);
                        break;
                }
            }
        }

        private enum EventKind
        {
            Added,
            Removed,
            Updated,
            Finished,
        }
    }
}