﻿#if UNITY_PURCHASES
using System;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using System.Linq;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Purchasing;

namespace Magify
{
    [SuppressMessage("ReSharper", "InconsistentNaming")]
    public class UnitySubscriptionModel
    {
        public string ProductId { [UsedImplicitly] get; }
        public UnityEngine.Purchasing.SubscriptionInfo SubscriptionInfo { get; }
        public string SubscriptionMetadata { get; }
        public string Receipt { get; }

        /// SubscriptionInfo API on new style
        public string GetProductId() => SubscriptionInfo.getProductId();

        public DateTime GetExpireDate() => SubscriptionInfo.getExpireDate();
        public DateTime GetPurchaseDate() => SubscriptionInfo.getPurchaseDate();
        public TimeSpan GetRemainingTime() => SubscriptionInfo.getRemainingTime();
        public TimeSpan GetIntroductoryPricePeriod() => SubscriptionInfo.getIntroductoryPricePeriod();
        public Result IsAutoRenewing() => SubscriptionInfo.isAutoRenewing();
        public Result IsSubscribed() => SubscriptionInfo.isSubscribed();
        public Result IsExpired() => SubscriptionInfo.isExpired();
        public Result IsCancelled() => SubscriptionInfo.isCancelled();

        public long GetIntroductoryPricePeriodCycles()
        {
            return Application.platform switch
            {
                RuntimePlatform.Android => getIntroductoryPricePeriodCyclesAndroid(),
                RuntimePlatform.IPhonePlayer => SubscriptionInfo.getIntroductoryPricePeriodCycles(),
                _ => -1
            };

            long getIntroductoryPricePeriodCyclesAndroid()
            {
                var receiptDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(Receipt);
                if (receiptDictionary.TryGetValue("Payload", out var payload))
                {
                    var payloadString = payload as string;
                    if (!string.IsNullOrEmpty(payloadString))
                    {
                        var payloadDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(payloadString);
                        if (payloadDictionary.TryGetValue("json", out var json))
                        {
                            var jsonString = json as string;
                            if (!string.IsNullOrEmpty(jsonString))
                            {
                                var jsonDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(jsonString);
                                if (jsonDictionary.TryGetValue("orderId", out var orderId))
                                {
                                    var orderIdString = orderId as string;
                                    if (!string.IsNullOrEmpty(orderIdString))
                                    {
                                        var dotsIndex = orderIdString.LastIndexOf("..", StringComparison.Ordinal);
                                        var iteration = dotsIndex == -1 ? 1 : int.Parse(orderIdString.Substring(dotsIndex + 2)) + 2;

                                        return iteration;
                                    }
                                }
                            }
                        }
                    }
                }

                return -1;
            }
        }

        public string GetIntroductoryPrice()
        {
            var introductoryPrice = SubscriptionInfo.getIntroductoryPrice();
            introductoryPrice = string.Concat(introductoryPrice.Where(symbol => Char.IsNumber(symbol) || symbol == ',' || symbol == '.'));

            return string.IsNullOrEmpty(introductoryPrice) ? null : introductoryPrice;
        }

        public Result IsIntroductoryPricePeriod()
        {
            return Application.platform switch
            {
                RuntimePlatform.Android => isIntroductoryPricePeriodAndroid(),
                RuntimePlatform.IPhonePlayer => SubscriptionInfo.isIntroductoryPricePeriod(),
                _ => Result.Unsupported
            };

            Result isIntroductoryPricePeriodAndroid()
            {
                if (GetIntroductoryPrice() != null)
                {
                    return GetIntroductoryPricePeriodCycles() == 1 ? Result.True : Result.False;
                }

                return Result.False;
            }
        }

        public Result IsFreeTrial()
        {
            return Application.platform switch
            {
                RuntimePlatform.Android => isFreeTrialAndroid(),
                RuntimePlatform.IPhonePlayer => SubscriptionInfo.isFreeTrial(),
                _ => Result.Unsupported
            };

            Result isFreeTrialAndroid()
            {
                if (HasTrialPeriod())
                {
                    return GetIntroductoryPricePeriodCycles() == 1 ? Result.True : Result.False;
                }

                return Result.False;
            }
        }

        /// <summary>
        /// For editor only
        /// </summary>
        public UnitySubscriptionModel(string productId)
        {
            ProductId = productId;
            SubscriptionInfo = new UnityEngine.Purchasing.SubscriptionInfo(productId);
        }

        /// <summary>
        /// For device only
        /// </summary>
        public UnitySubscriptionModel(Product product, string subscriptionMetadata, UnityEngine.Purchasing.SubscriptionInfo subscriptionInfo)
        {
            ProductId = product.definition.storeSpecificId;
            Receipt = product.receipt;
            SubscriptionInfo = subscriptionInfo;
            SubscriptionMetadata = subscriptionMetadata;
        }

        public bool HasTrialPeriod()
        {
            return HasTrialPeriod(SubscriptionMetadata);
        }

        public static bool HasTrialPeriod(string subscriptionMetadata)
        {
            if (Application.isEditor || string.IsNullOrEmpty(subscriptionMetadata))
            {
                return false;
            }

            var receiptDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(subscriptionMetadata);

            return Application.platform switch
            {
                RuntimePlatform.Android => hasTrialAndroid(),
                RuntimePlatform.IPhonePlayer => hasTrialIOS(),
                _ => false
            };

            bool hasTrialAndroid()
            {
                if (receiptDictionary.TryGetValue("freeTrialPeriod", out var freeTrialPeriod))
                {
                    return !string.IsNullOrEmpty((string)freeTrialPeriod);
                }

                return false;
            }

            bool hasTrialIOS()
            {
                if (receiptDictionary.TryGetValue("introductoryPrice", out var introductoryPrice))
                {
                    var introductoryPriceString = introductoryPrice as string;

                    return !string.IsNullOrEmpty(introductoryPriceString) && Int32.Parse(introductoryPriceString) == 0;
                }

                return false;
            }
        }

        public SubscriptionPeriodModel GetSubscriptionPeriod()
        {
            return GetSubscriptionPeriod(SubscriptionMetadata);
        }

        public static SubscriptionPeriodModel GetSubscriptionPeriod(string subscriptionMetadata)
        {
            if (Application.isEditor || string.IsNullOrEmpty(subscriptionMetadata))
            {
                return default;
            }

            var receiptDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(subscriptionMetadata);

            return Application.platform switch
            {
                RuntimePlatform.Android => getPeriodAndroid(),
                RuntimePlatform.IPhonePlayer => getPeriodIOS(),
                _ => default
            };

            SubscriptionPeriodModel getPeriodAndroid()
            {
                if (receiptDictionary.TryGetValue("subscriptionPeriod", out var subscriptionPeriod))
                {
                    return new SubscriptionPeriodModel((string)subscriptionPeriod);
                }

                return default;
            }

            SubscriptionPeriodModel getPeriodIOS()
            {
                if (receiptDictionary.TryGetValue("subscriptionPeriodUnit", out var subscriptionPeriodUnit) &&
                    receiptDictionary.TryGetValue("subscriptionNumberOfUnits", out var subscriptionNumberOfUnits))
                {
                    var subscriptionPeriodUnitString = (string)subscriptionPeriodUnit;
                    var subscriptionNumberOfUnitsString = (string)subscriptionNumberOfUnits;

                    if (string.IsNullOrEmpty(subscriptionPeriodUnitString) || string.IsNullOrEmpty(subscriptionNumberOfUnitsString))
                    {
                        return default;
                    }

                    var periodType = (SubscriptionPeriodType)int.Parse(subscriptionPeriodUnitString);
                    var numberPeriods = int.Parse(subscriptionNumberOfUnitsString);

                    return new SubscriptionPeriodModel(periodType, numberPeriods);
                }

                return default;
            }
        }

        public string GetPrice()
        {
            return GetPrice(SubscriptionMetadata);
        }

        public static string GetPrice(string subscriptionMetadata)
        {
            if (Application.isEditor || string.IsNullOrEmpty(subscriptionMetadata))
            {
                return default;
            }

            var receiptDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(subscriptionMetadata);

            return Application.platform switch
            {
                RuntimePlatform.Android => getTrialPeriodAndroid(),
                RuntimePlatform.IPhonePlayer => getTrialPeriodIOS(),
                _ => string.Empty
            };

            string getTrialPeriodAndroid()
            {
                if (receiptDictionary.TryGetValue("price", out var price))
                {
                    return (string)price;
                }

                return string.Empty;
            }

            string getTrialPeriodIOS()
            {
                if (receiptDictionary.TryGetValue("localizedPriceString", out var localizedPriceString))
                {
                    return (string)localizedPriceString;
                }

                return string.Empty;
            }
        }

        public SubscriptionPeriodModel GetTrialPeriod()
        {
            return GetTrialPeriod(SubscriptionMetadata);
        }

        public static SubscriptionPeriodModel GetTrialPeriod(string subscriptionMetadata)
        {
            if (Application.isEditor || string.IsNullOrEmpty(subscriptionMetadata))
            {
                return default;
            }

            var receiptDictionary = (Dictionary<string, object>)MiniJson.JsonDecode(subscriptionMetadata);

            return Application.platform switch
            {
                RuntimePlatform.Android => getTrialPeriodAndroid(),
                RuntimePlatform.IPhonePlayer => getTrialPeriodIOS(),
                _ => default
            };

            SubscriptionPeriodModel getTrialPeriodAndroid()
            {
                if (receiptDictionary.TryGetValue("freeTrialPeriod", out var freeTrialPeriod))
                {
                    return new SubscriptionPeriodModel((string)freeTrialPeriod).ToWeeksToDays();
                }

                return default;
            }

            SubscriptionPeriodModel getTrialPeriodIOS()
            {
                if (receiptDictionary.TryGetValue("unit", out var unit) &&
                    receiptDictionary.TryGetValue("numberOfUnits", out var numberOfUnits))
                {
                    var unitString = (string)unit;
                    var numberOfUnitsString = (string)numberOfUnits;

                    if (string.IsNullOrEmpty(unitString) || string.IsNullOrEmpty(numberOfUnitsString))
                    {
                        return default;
                    }

                    var periodType = (SubscriptionPeriodType)int.Parse(unitString);
                    var numberPeriods = int.Parse(numberOfUnitsString);

                    return new SubscriptionPeriodModel(periodType, numberPeriods).ToWeeksToDays();
                }

                return default;
            }
        }

        #region Obsolete

        /// SubscriptionInfo API
        [Obsolete("GetIntroductoryPricePeriodCycles()", true)]
        public long getIntroductoryPricePeriodCycles() => GetIntroductoryPricePeriodCycles();

        [Obsolete("GetProductId()", true)]
        public string getProductId() => SubscriptionInfo.getProductId();

        [Obsolete("GetIntroductoryPrice()", true)]
        public string getIntroductoryPrice() => GetIntroductoryPrice();

        [Obsolete("GetExpireDate()", true)]
        public DateTime getExpireDate() => SubscriptionInfo.getExpireDate();

        [Obsolete("GetPurchaseDate()", true)]
        public DateTime getPurchaseDate() => SubscriptionInfo.getPurchaseDate();

        [Obsolete("GetSubscriptionPeriod()", true)]
        public TimeSpan getSubscriptionPeriod() => SubscriptionInfo.getSubscriptionPeriod();

        [Obsolete("GetRemainingTime()", true)]
        public TimeSpan getRemainingTime() => SubscriptionInfo.getRemainingTime();

        [Obsolete("GetIntroductoryPricePeriod()", true)]
        public TimeSpan getIntroductoryPricePeriod() => SubscriptionInfo.getIntroductoryPricePeriod();

        [Obsolete("IsIntroductoryPricePeriod()", true)]
        public Result isIntroductoryPricePeriod() => IsIntroductoryPricePeriod();

        [Obsolete("IsAutoRenewing()", true)]
        public Result isAutoRenewing() => SubscriptionInfo.isAutoRenewing();

        [Obsolete("IsSubscribed()", true)]
        public Result isSubscribed() => SubscriptionInfo.isSubscribed();

        [Obsolete("IsExpired()", true)]
        public Result isExpired() => SubscriptionInfo.isExpired();

        [Obsolete("IsCancelled()", true)]
        public Result isCancelled() => SubscriptionInfo.isCancelled();

        [Obsolete("IsFreeTrial()", true)]
        public Result isFreeTrial() => IsFreeTrial();

        #endregion
    }
}
#endif