using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Rx;
using UnityEngine;
using UnityEngine.Pool;
using UnityEngine.UI;

namespace Magify
{
    public class PopupBase<TArgs> : MonoBehaviour, IPopupBase<TArgs>
        where TArgs : PopupArgs
    {
        [SerializeField]
        private Transform _content;
        [SerializeField]
        private Easing.Function _showFunction = Easing.Function.ElasticEaseOut;
        [SerializeField]
        private float _showTime = 0.3f;
        [SerializeField]
        private Easing.Function _hideFunction = Easing.Function.Linear;
        [SerializeField]
        private float _hideTime = 0.15f;
        [SerializeField]
        private Button[] _closeButtons = Array.Empty<Button>();

        private CompositeDisposable _disposables;
        private CancellationTokenSource _showCancellationTokenSource;
        private CancellationTokenSource _hideCancellationTokenSource;

        public event Action OnHideRequested;

        protected virtual void PrepareForShow(TArgs arguments, CompositeDisposable disposables)
        {
            MagifyService.Instance.Advertiser.TemporaryDisableBanner().AddTo(_disposables);
        }

        public async UniTask ShowAsync(TArgs arguments, CancellationToken cancellationToken)
        {
            if (_showCancellationTokenSource != null)
            {
                throw new Exception("Popup already shown (or being showing)");
            }
            _showCancellationTokenSource = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            _disposables = GenericPool<CompositeDisposable>.Get();

            PrepareForShow(arguments, _disposables);

            foreach (var closeButton in _closeButtons)
            {
                if (arguments.CloseButtonDelay > 0)
                {
                    closeButton.gameObject.SetActive(false);
                    Observable.Timer(TimeSpan.FromSeconds(arguments.CloseButtonDelay))
                        .Subscribe(_ => { closeButton.gameObject.SetActive(true); })
                        .AddTo(_disposables);
                }
                else
                {
                    closeButton.gameObject.SetActive(true);
                }

                closeButton.SubscribeOnClick(RequestHiding)
                    .AddTo(_disposables);
            }

            await ShowAnimationAsync(_showCancellationTokenSource.Token);
        }

        public async UniTask HideAsync(CancellationToken cancellationToken)
        {
            if (_showCancellationTokenSource == null || _hideCancellationTokenSource != null)
            {
                throw new Exception("Popup already hidden (or being hiding)");
            }
            _hideCancellationTokenSource = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);

            await HideAnimationAsync(_hideCancellationTokenSource.Token);
            RequestHiding();
        }

        public void Dispose()
        {
            _disposables.Clear();
            GenericPool<CompositeDisposable>.Release(_disposables);
            _showCancellationTokenSource?.Cancel();
            _hideCancellationTokenSource?.Cancel();
            _disposables = null;
            _showCancellationTokenSource = null;
            _hideCancellationTokenSource = null;
        }

        protected void RequestHiding()
        {
            OnHideRequested?.Invoke();
        }

        protected virtual UniTask ShowAnimationAsync(CancellationToken cancellationToken)
        {
            // TODO flexible logic for Show animation
            return EaseScale(0, 1, _showFunction, _showTime, cancellationToken);
        }

        protected virtual UniTask HideAnimationAsync(CancellationToken cancellationToken)
        {
            // TODO flexible logic for Hide animation
            return EaseScale(1, 0, _hideFunction, _hideTime, cancellationToken);
        }

        protected async UniTask EaseScale(float from, float to, Easing.Function function, float processTime, CancellationToken cancellationToken)
        {
            var time = 0.0f;
            while (time < processTime)
            {
                var t = time / processTime;
                t = Mathf.Clamp01(t);
                t = Easing.Evaluate(function, t);
                var scale = Mathf.Lerp(from, to, t);
                _content.localScale = new Vector3(scale, scale, scale);
                await UniTask.Yield(cancellationToken);
                time += Time.deltaTime;
            }

            _content.localScale = new Vector3(to, to, to);
        }
    }
}