using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    public partial class MagifyService
    {
        private const string PrefsName = "service.bin";
#if MAGIFY_ADVANCED
        internal const bool IsAdvanced = true;
#else
        internal const bool IsAdvanced = false;
#endif

        [CanBeNull]
        public static MagifyService Instance { get; private set; }

#pragma warning disable CS0162 // Unreachable code detected
        /// <summary>
        /// You have to call this method before using <see cref="MagifyService"/>.<see cref="Instance"/>
        /// </summary>
        /// <remarks>
        /// Don't forget to call <see cref="Terminate"/> method when you don't need <see cref="MagifyService"/> anymore.
        /// </remarks>
        /// <exception cref="MagifyAlreadyInitializedException">
        /// If you try to initialize <see cref="MagifyService"/> more than once.
        /// </exception>
        /// <exception cref="NoAppNameForMagifyInitializationException">
        /// If you try to initialize <see cref="MagifyService"/> without <see cref="MagifySettings.NameGP"/> or <see cref="MagifySettings.NameIOS"/> in the settings.
        /// </exception>
        /// <exception cref="MagifyClientIdCannotBeChangedException">
        /// If you are attempting to pass <see cref="MagifyServiceArgs.CustomClientId"/> for the second+ time, and it's now different from the previous one.
        /// </exception>
        [NotNull]
        public static MagifyService Initialize([NotNull] MagifySettings settings, [CanBeNull] MagifyServiceArgs args)
        {
            if (Instance != null)
            {
                throw new MagifyAlreadyInitializedException();
            }
            // Prepare prefs
            var storagePath = Path.Combine(PackageInfo.PersistentPath, PrefsName);
            var prefs = ServicePrefs.Create(storagePath);
            var offerNativeEventsAccumulator = IsAdvanced ? new LimitedTimeOfferNativeEventsAccumulator() : null;
            var debugConfig = default(MagifyDebugConfig);
#if MAGIFY_VERBOSE_LOGGING || MAGIFY_DEBUG || UNITY_EDITOR
            debugConfig = new MagifyDebugConfig
            {
                CustomAndroidId = args?.CustomAndroidId,
                CustomBuildNumber = args?.CustomBuildNumber,
                CustomAppVersion = args?.CustomAppVersion,
            };
#endif
            MagifyManager.Initialize(new MagifyConfig
            {
#if !MAGIFY_3_OR_NEWER
                EditorMoq = args?.EditorMoq,
#endif
                PresenterSettings = settings,
                AppNameGP = settings.NameGP,
                AppNameIOS = settings.NameIOS,
                ConfigPathGP = settings.ConfigGP,
                ConfigPathIOS = settings.ConfigIOS,
                CustomClientId = args?.CustomClientId,
                ClientStateConfig = args?.ClientStateConfig ?? new ClientStateConfig(),
                UseAdvancedVersion = IsAdvanced,
                IsDeveloperMode = args?.IsDeveloperMode ?? false,
                EditorDevice = args?.EditorConfigMode is RuntimePlatform.IPhonePlayer
                    ? EditorDevice.GetDefaultIOSDevice()
                    : EditorDevice.GetDefaultAndroidDevice(),
                Environment = args?.Environment ?? Magify.Environment.Staging,
                RemoteConfigScopes = args?.RemoteConfigScopes ?? ConfigScope.None,
                SyncStateEnabled = args?.SyncStateEnabled,
                IsAutoRestoreStateEnabled = args?.IsAutoRestoreStateEnabled ?? true,
                SkipClientIdFromCloudLoading = args?.SkipClientIdFromCloudLoading ?? false,
                ClientIdFromCloudLoadingTimeout = args?.ClientIdFromCloudLoadingTimeout ?? 30,
                SubscriptionStatus = prefs.SubscriptionStatus.Value,
                IsGdprApplied = args?.IsGdprApplied,
                IsAttAuthorized = args?.IsAttAuthorized,
                IsLoggingEnabled = args?.IsLoggingEnabled ?? true,
            }, debugConfig);
            offerNativeEventsAccumulator?.StopEventsCatching();
            Instance = new MagifyService(settings, args, prefs);
#if MAGIFY_ADVANCED
            if (IsAdvanced)
            {
                Instance.Offers.HandleOfferEventsInNextFrameStart(offerNativeEventsAccumulator).Forget();
            }
#endif
#if !MAGIFY_3_OR_NEWER
            MagifyManager.TrackAppLaunch();
#endif
            return Instance;
        }
#pragma warning restore CS0162 // Unreachable code detected

        /// <summary>
        /// This method allows you to terminate <see cref="MagifyService"/> and free all resources.
        /// </summary>
        public static void Terminate()
        {
            if (Instance == null) return;
            Instance.ShutDown();
            Instance = null;
        }

        /// <inheritdoc cref="WaitForInitialized(System.Threading.CancellationToken, Cysharp.Threading.Tasks.PlayerLoopTiming)"/>
        [NotNull, ItemNotNull]
        public static UniTask<MagifyService> WaitForInitialized(PlayerLoopTiming timing = PlayerLoopTiming.Update) => WaitForInitialized(CancellationToken.None, timing);

        /// <summary>
        /// This method allows you to wait until <see cref="MagifyService"/> is initialized.
        /// </summary>
        /// <returns>
        /// Initialized <see cref="MagifyService"/> instance
        /// </returns>
        public static async UniTask<MagifyService> WaitForInitialized(CancellationToken cancellationToken, PlayerLoopTiming timing = PlayerLoopTiming.Update)
        {
            await UniTask.WaitUntil(() => Instance != null, timing, cancellationToken: cancellationToken);
            return Instance;
        }

        internal static void ThrowIfNotReady(string source = "")
        {
            if (Instance == null)
            {
                throw new MagifyServiceNotInitializedException(source);
            }
        }
    }
}