﻿using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using UnityEngine;

namespace Magify
{
    internal class RewardProductObtainer : ProductObtainer<RewardProduct>, IProductObtainer
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly AdvertiserService _advertiser;

        public RewardProductObtainer([NotNull] AdvertiserService advertiser)
        {
            _advertiser = advertiser;
        }

        public bool ObtainUnderSpin => true;

        public override bool CanObtainProduct(ProductDef product) => base.CanObtainProduct(product) && _advertiser.Mediator != null;

        public bool NeedPrepareProduct(ProductDef product)
        {
            return !_advertiser.Mediator.IsRewardedVideoReady;
        }

        public async UniTask<ProductPrepareResult> PrepareProductAsync(ProductDef product, CancellationToken cancellationToken)
        {
            _logger.Log("Load RewardVideo, show loader panel and wait for result");
            var loadResult = await _advertiser.LoadRewardVideoAsync(cancellationToken);
            _logger.Log($"Load RewardVideo finished with result: {loadResult}");
            return ProductPrepareResult.FromAdLoadResult(loadResult);
        }

        private protected override async UniTask<ProductObtainResult> ObtainProductAsync(RewardProduct product, CampaignRequest request, CancellationToken cancellationToken)
        {
            var spot = request.Event;
            var campaign = request.Campaign;

            _logger.Log("Show RewardVideo and wait for result");
            var mediatorResult = await _advertiser.ShowRewardedVideoAsync(
                spot,
                impression =>
                {
                    _logger.Log($"RewardedVideo - TrackClickFor: {nameof(impression)}={JsonFacade.SerializeObject(impression)}");
                    request.TrackAdsClick(product);
                },
                cancellationToken);
            _logger.Log($"Show RewardVideo finished with result: {mediatorResult}");

            switch (mediatorResult.State)
            {
                case AdShowState.Showed:
                    _logger.Log("RewardVideo successfully showed");
                    // TODO reward counting
                    // _profile.RewardedProducts.TryAdd(product.Id, new ReactiveProperty<int>());
                    // var count = ++_profile.RewardedProducts[product.Id].Value;
                    //
                    // var origin = (RewardProduct)product;
                    // if (origin.Count == 0 || count % origin.Count == 0)
                    // {
                    //     _callbacks.RiseOnRewardGranted();
                    //     _wallet.BeginTransactionsCaching();
                    //     _purchaseHandler?.ApplyPurchase(new PurchaseInfo(product, spot, campaign));
                    //     _wallet.FinishTransactionsCaching();
                    //     Log($"Track {nameof(TrackRewardGranted)} for product {product.Id} during {campaign.Name} campaign triggered by {spot}");
                    //     TrackRewardGranted(product.Id);
                    //     productObtainResult = new ProductObtainResult(ProductObtainResult.ResultType.Bought, product, spot, campaign);
                    // }
                    // else
                    // {
                    //     productObtainResult = new ProductObtainResult(ProductObtainResult.ResultType.Undefined, product, spot, campaign);
                    // }
                    request.TrackAdsShow(mediatorResult.Impression);
                    return new ProductObtainResult(product, request);
                case AdShowState.Closed:
                    _logger.Log("RewardVideo was closed");
                    return new ProductObtainResult(product, request, ProductObtainFailReason.UserCancelled);
                default:
                    _logger.Log($"Failed to show RewardVideo. FailType={mediatorResult.State}. Error: {mediatorResult.Error}");
                    return new ProductObtainResult(product, request, ProductObtainFailReason.Unknown);
            }
        }
    }
}