﻿using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;

namespace Magify
{
    internal class InAppProductObtainer : IProductObtainer
    {
        private readonly PurchaserService _purchaser;

        public bool ObtainUnderSpin => true;

        public InAppProductObtainer([NotNull] PurchaserService purchaser)
        {
            _purchaser = purchaser;
        }

        public bool CanObtainProduct(ProductDef product) => product is InAppProduct or SubscriptionProduct && _purchaser.InAppStore != null;

        public bool NeedPrepareProduct(ProductDef product)
        {
            return !_purchaser.InAppStore.IsProductReady(product);
        }

        public UniTask<ProductPrepareResult> PrepareProductAsync(ProductDef product, CancellationToken cancellationToken)
        {
            return _purchaser.PrepareProductAsync(product, cancellationToken);
        }

        public async UniTask<ProductObtainResult> ObtainProductAsync(ProductDef product, CampaignRequest request, CancellationToken cancellationToken)
        {
            var purchaseResult = product switch
            {
                InAppProduct inAppProduct => await _purchaser.PurchaseProductAsync(inAppProduct, cancellationToken),
                SubscriptionProduct subscriptionProduct => await _purchaser.PurchaseProductAsync(subscriptionProduct, cancellationToken),
                _ => throw new ArgumentOutOfRangeException($"{nameof(product)} has type {product.GetType()} but must be {nameof(InAppProduct)} or {nameof(SubscriptionProduct)}")
            };
            return purchaseResult.FailReason == null
                ? new ProductObtainResult(product, request, purchaseResult.Info)
                : new ProductObtainResult(product, purchaseResult.FailReason.Value);
        }
    }
}