using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    internal class RateReviewCampaignHandler : CampaignHandler<RateReviewCampaign>
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly ServicePrefs _prefs;
        private readonly MagifySettings _settings;
        private readonly PopupsHandler _popups;

        internal RateReviewCampaignHandler(MagifySettings settings, ServicePrefs prefs, PopupsHandler popups)
        {
            _popups = popups;
            _settings = settings;
            _prefs = prefs;
        }

        public override async UniTask<CampaignResult> HandleCampaignAsync(CampaignRequest request, CancellationToken cancellationToken)
        {
            _logger.Log($"Trying to ask rate application. SupportNativeRateReview={SupportNativeRateReview()}; NativeRateUsShownBefore={_prefs.NativeRateUsShown.Value}");
            if (SupportNativeRateReview() && !_prefs.NativeRateUsShown.Value)
            {
                _prefs.NativeRateUsShown.Value = true;
                try
                {
                    request.TrackShow();
                    _logger.Log("Calling native RateReview flow");
                    request.TrackClick();
                    var nativeResult = await RequestAppReview(cancellationToken);
                    if (cancellationToken.IsCancellationRequested) return CampaignResult.Aborted;
                    _logger.Log($"Native RateReview flow returned: {nativeResult}");
                    return nativeResult ? CampaignResult.Applied : CampaignResult.Failed;
                }
                catch (OperationCanceledException)
                {
                    return CampaignResult.Aborted;
                }
            }

            var url = Application.platform switch
            {
                RuntimePlatform.Android => string.IsNullOrEmpty(_settings.RateUsLinkAndroid) ? null : _settings.RateUsLinkAndroid,
                RuntimePlatform.IPhonePlayer => string.IsNullOrEmpty(_settings.RateUsLinkIOS) ? null : _settings.RateUsLinkIOS,
                RuntimePlatform.WindowsEditor or RuntimePlatform.LinuxEditor or RuntimePlatform.OSXEditor => "https://www.appsyoulove.com/",
                _ => null
            };

            if (string.IsNullOrEmpty(url))
            {
                _logger.LogError($"Failed to build url. Check section 'Bundle Info' in {nameof(MagifySettings)}");
                request.TrackShowFailed("Invalid link");
                return CampaignResult.Failed;
            }

            _logger.Log($"Open RateReviewPopup to aks user is he ready to rate application");
            var args = new PopupArgs
            {
                CloseButtonDelay = (float)_settings.RateReviewCloseButtonDelay.Value
            };

            var clicked = false;
            var showResult = await _popups.ShowRateReview(args, request.TrackShow, cancellationToken, async () =>
            {
                request.TrackClick();
                _logger.Log($"Opening url to store: {url}");
                var restoreType = await ApplicationHelper.CallAndWaitForApplicationRestore(() => Application.OpenURL(url), cancellationToken);
                if (cancellationToken.IsCancellationRequested) return ClickHandleResult.Canceled;
                clicked = true;
                _logger.Log($"The application went out of the background after opening store. Restore type is {restoreType}");
                return ClickHandleResult.ClosePopup;
            });
            cancellationToken.ThrowIfCancellationRequested();

            if (showResult == PopupShowResult.Showed) return clicked ? CampaignResult.Applied : CampaignResult.Declined;

            _logger.LogError("Failed to show RateReview popup");
            request.TrackShowFailed("Failed to show RateReview popup");
            return CampaignResult.Failed;
        }

        private bool SupportNativeRateReview()
        {
            return Application.platform switch
            {
#if GOOGLE_PLAY_REVIEW
                RuntimePlatform.Android => true,
#endif
                RuntimePlatform.IPhonePlayer => true,
                _ => false
            };
        }

        internal static async UniTask<bool> RequestAppReview(CancellationToken cancellationToken)
        {
#if UNITY_IOS
            var result = UnityEngine.iOS.Device.RequestStoreReview();
            await UniTask.Delay(250, true, cancellationToken: cancellationToken);
            return result;
#elif UNITY_ANDROID && GOOGLE_PLAY_REVIEW
            var reviewManager = new Google.Play.Review.ReviewManager();
            var requestFlowOperation = reviewManager.RequestReviewFlow();
            await requestFlowOperation.ToUniTask(cancellationToken: cancellationToken);
            if (requestFlowOperation.Error != Google.Play.Review.ReviewErrorCode.NoError)
            {
                return false;
            }
            var playReviewInfo = requestFlowOperation.GetResult();
            var launchFlowOperation = reviewManager.LaunchReviewFlow(playReviewInfo);
            await UniTask.WaitUntil(() => launchFlowOperation.IsDone, cancellationToken: cancellationToken);
            return launchFlowOperation.Error == Google.Play.Review.ReviewErrorCode.NoError;
#else
            await UniTask.Yield();
            return false;
#endif
        }
    }
}