using System;
using System.Collections.Generic;
using System.Linq;
using Cysharp.Threading.Tasks;
using Magify.Rx;
using UnityEngine;
using UnityEngine.Pool;

namespace Magify
{
    public class DelayedCampaignRegistry
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly List<CampaignRequest> _delayedCampaigns = new();
        private readonly Subject<(CampaignRequest, CampaignResult)> _onDelayedCampaignFinished = new();

        public IObservable<(CampaignRequest Request, CampaignResult Result)> OnDelayedCampaignFinished => _onDelayedCampaignFinished;

        public bool Any => _delayedCampaigns.Any();

        public CampaignRequest GetFirstCampaignByType(CampaignType type)
        {
            return _delayedCampaigns.FirstOrDefault(c => c.Campaign.Type == type);
        }

        public void DelayCampaign(CampaignRequest request, UniTask<CampaignResult> task)
        {
            DelayCampaignAsync(request, task).Forget();
        }

        private async UniTaskVoid DelayCampaignAsync(CampaignRequest request, UniTask<CampaignResult> task)
        {
            var campaign = request.Campaign;
            _logger.Log($"Delay campaign {campaign.Name} ({campaign.Type}) for event {request.Event}");
            if (HasCampaignWithType(campaign.Type))
            {
                _logger.LogError($"There is existing delayed campaign with type {campaign.Type}!");
                for (var i = _delayedCampaigns.Count; i >= 0; i--)
                {
                    if (_delayedCampaigns[i].Campaign.Type == campaign.Type)
                    {
                        _delayedCampaigns.RemoveAt(i);
                    }
                }
            }
            _delayedCampaigns.Add(request);

            var result = CampaignResult.None;
            try
            {
                result = await task;
            }
            catch (OperationCanceledException)
            {
                _logger.Log($"Delayed campaign {campaign.Name} was cancelled");
                result = CampaignResult.Aborted;
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                request.TrackShowFailed($"Failed with exception {e.GetType().Name}: {e.Message}");
                result = CampaignResult.Failed;
            }
            finally
            {
                _logger.Log($"DelayedCampaigns campaign {campaign.Name} ({campaign.Type}) finished with result {result}");
                _delayedCampaigns.Remove(request);
                request.Disposables.Clear();
                GenericPool<CompositeDisposable>.Release(request.Disposables);
                _onDelayedCampaignFinished.OnNext((request, result));
            }
        }

        private bool HasCampaignWithType(CampaignType type)
        {
            return _delayedCampaigns.FirstOrDefault(c => c.Campaign.Type == type) != null;
        }
    }
}