﻿using System.Collections.Generic;
using System.Linq;
using JetBrains.Annotations;
using UnityEngine;
using UnityEngine.Pool;
using static Magify.MagifyManager;

namespace Magify
{
    public class CampaignsProvider
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly Dictionary<string, Dictionary<string, object>> _globalParams = new();
        private readonly Dictionary<string, object> _globalUniversalParams = new();
        private readonly List<ICampaignHandler> _embeddedHandlers = new();
        private readonly List<ICampaignHandler> _externalHandlers = new();

        internal void RegisterEmbeddedCampaignHandler(ICampaignHandler handler)
        {
            _embeddedHandlers.Add(handler);
        }

        public void RegisterCampaignHandler(ICampaignHandler handler)
        {
            _externalHandlers.Add(handler);
        }

        internal ICampaignHandler GetHandlerFor(CampaignRequest request)
        {
            return _externalHandlers.FirstOrDefault(c => c.CanHandleCampaign(request)) ?? _embeddedHandlers.FirstOrDefault(c => c.CanHandleCampaign(request));
        }

        [CanBeNull]
        public ICampaign GetCampaign(string @event, Dictionary<string, object> @params = null)
        {
            return GetCampaignToHandle(@event, @params, true);
        }

        public bool IsCampaignAvailable(string campaignName, string @event, Dictionary<string, object> @params = null)
        {
            if (Application.isEditor) return true;
            var mergedParams = CombineParams(@event, @params);
            var result = MagifyManager.IsCampaignAvailable(campaignName, @event, mergedParams);
            if (mergedParams != null)
            {
                DictionaryPool<string, object>.Release(mergedParams);
            }
            return result;
        }

        [CanBeNull]
        internal ICampaign GetCampaignToHandle(string @event, Dictionary<string, object> @params, bool silent = false)
        {
            var flags = silent ? CampaignRequestFlags.Silent : CampaignRequestFlags.None;
            flags |= CampaignRequestFlags.ReplaceCampaignTypeWithString;
            return GetCampaignToHandle(@event, @params, flags);
        }

        [CanBeNull]
        private ICampaign GetCampaignToHandle(string @event, Dictionary<string, object> @params, CampaignRequestFlags flags)
        {
            var mergedParams = CombineParams(@event, @params);
            var campaign = CampaignFor(@event, flags, mergedParams);
            if (mergedParams != null)
            {
                DictionaryPool<string, object>.Release(mergedParams);
            }
            return campaign;
        }

        /// <summary>
        /// Set the parameter value for the target event (spot). This parameter will be added to all following campaign requests.
        /// </summary>
        public void SetParam(string @event, string param, object value)
        {
            _logger.Log($"Adding global param {param}: {value} for spot {@event}");
            if (!_globalParams.TryGetValue(@event, out var parameters))
            {
                parameters = DictionaryPool<string, object>.Get();
                _globalParams[@event] = parameters;
            }
            parameters[param] = value;
        }

        /// <summary>
        /// Set the parameter value for the target event (spot). This parameter will be added to all following campaign requests.
        /// </summary>
        public void SetParams(string @event, IReadOnlyDictionary<string, object> @params)
        {
            @params.ForEach(param => SetParam(@event, param.Key, param.Value));
        }

        /// <summary>
        /// Set the parameter value for all events (spots). This parameter will be added to all following campaign requests.
        /// </summary>
        public void SetUniversalParam(string param, object value)
        {
            _logger.Log($"Adding global param {param}: {value} for all spots");
            _globalUniversalParams[param] = value;
        }

        /// <summary>
        /// Set the parameter value for all events (spots). This parameter will be added to all following campaign requests.
        /// Set param value for all events (spots)
        /// </summary>
        public void SetUniversalParams(IReadOnlyDictionary<string, object> @params)
        {
            @params.ForEach(param => SetUniversalParam(param.Key, param.Value));
        }

        /// <summary>
        /// Remove the parameter for the target event (spot). This parameter will no longer be added to all following campaign requests.
        /// </summary>
        public void RemoveParam(string @event, string param)
        {
            _logger.Log($"Removing param '{param}' for spot {@event}");
            if (!_globalParams.TryGetValue(@event, out var parameters)) return;
            parameters.Remove(param);
            if (parameters.Count == 0)
            {
                _globalParams.Remove(@event);
                DictionaryPool<string, object>.Release(parameters);
            }
        }

        /// <summary>
        /// Remove all parameters for the target event (spot). All parameters will no longer be added to all following campaign requests.
        /// </summary>
        public void RemoveAllParams(string @event)
        {
            _logger.Log($"Removing all params for spot {@event}");
            if (!_globalParams.TryGetValue(@event, out var parameters)) return;
            _globalParams.Remove(@event);
            DictionaryPool<string, object>.Release(parameters);
        }

        /// <summary>
        /// Remove the universal parameter. This parameter will no longer be added to all following campaign requests.
        /// </summary>
        public void RemoveUniversalParam(string param)
        {
            _logger.Log($"Removing universal param '{param}'");
            _globalUniversalParams.Remove(param);
        }

        /// <summary>
        /// Remove all universal parameters. All parameters will no longer be added to all following campaign requests.
        /// </summary>
        public void RemoveAllUniversalParams()
        {
            _logger.Log($"Removing all universal params");
            _globalUniversalParams.Clear();
        }

        private Dictionary<string, object> CombineParams(string @event, Dictionary<string, object> @params)
        {
            _globalParams.TryGetValue(@event, out var globalParams);
            if ((@params == null || @params.Count == 0) && globalParams == null)
            {
                return null;
            }
            var mergedParams = DictionaryPool<string, object>.Get();
            mergedParams.EnsureCapacity(globalParams?.Count ?? 0 + @params.Count + _globalUniversalParams.Count);

            globalParams?.ForEach(param => mergedParams[param.Key] = param.Value);
            @params?.ForEach(param => mergedParams[param.Key] = param.Value);
            _globalUniversalParams.ForEach(param => mergedParams[param.Key] = param.Value);
            return mergedParams;
        }
    }
}