using System;
using System.Runtime.CompilerServices;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class BaseAdvertiserService : IDisposable
    {
        [NotNull]
        private protected static readonly MagifyLogger Logger = MagifyLogger.Get(MagifyService.LogScope);

        [NotNull]
        private readonly MagifySettings _settings;
        [NotNull]
        private readonly ServicePrefs _prefs;
        [CanBeNull]
        private readonly CampaignsProvider _campaigns;
        [NotNull]
        private readonly INetworkStatusProvider _network;
        [NotNull]
        private protected readonly AdPreloader AdPreloader;

        [NotNull]
        protected readonly CompositeDisposable Disposables = new();

        [NotNull]
        private readonly ReactiveProperty<int> _bannerDeactivationsCounter = new(0);
        [NotNull]
        protected readonly ReactiveProperty<bool> BannerRequired = new(false);
        [NotNull]
        protected readonly ReactiveProperty<bool> BannerActive = new(false);

        [NotNull]
        private readonly Subject<Unit> _onBeforeShowRewardVideo = new();
        [NotNull]
        private readonly Subject<Unit> _onAfterShowRewardVideo = new();

        [NotNull]
        private readonly Subject<Unit> _onBeforeShowInterstitialVideo = new();
        [NotNull]
        private readonly Subject<Unit> _onAfterShowInterstitialVideo = new();

        protected readonly bool IsBannerCampaignRequired;

        [NotNull]
        public IReactiveProperty<bool> AdPreloadEnabled => AdPreloader.IsEnabled;

        /// <summary>
        /// Determines if a banner is required by application
        /// </summary>
        [NotNull]
        public IReactiveProperty<bool> IsBannerRequired => BannerRequired;

        /// <summary>
        /// Determines if a banner currently visible for player.
        /// It can be false even when <see cref="IsBannerRequired"/> is true because someone could request a temporary disable using <see cref="TemporaryDisableBanner"/>
        /// </summary>
        [NotNull]
        public IReadOnlyReactiveProperty<bool> IsBannerActive => BannerActive;

        // ReSharper disable InconsistentNaming
        [NotNull]
        public IObservable<Unit> OnBeforeShowRewardVideo => _onBeforeShowRewardVideo;
        [NotNull]
        public IObservable<Unit> OnAfterShowRewardVideo => _onAfterShowRewardVideo;
        [NotNull]
        public IObservable<Unit> OnBeforeShowInterstitialVideo => _onBeforeShowInterstitialVideo;
        [NotNull]
        public IObservable<Unit> OnAfterShowInterstitialVideo => _onAfterShowInterstitialVideo;
        // ReSharper restore InconsistentNaming

        [CanBeNull]
        public virtual IMinimalAdsMediator MinimalMediator { get; protected set; }

        [CanBeNull]
        public virtual IAdsMediator Mediator
        {
            get => CheckMediator($"get {nameof(Mediator)}") ? (IAdsMediator)MinimalMediator : null;
            protected set => MinimalMediator = value;
        }

        [MethodImpl(MethodImplOptions.AggressiveInlining)]
        private bool CheckMediator([NotNull] string reason)
        {
            if (MinimalMediator is IAdsMediator)
                return true;
            Logger.LogError(MinimalMediator == null
                ? $"You are trying to {reason} but you didn't set the mediator. Please, use {nameof(MagifyService.SetAdsMediator)} method to set it."
                : $"You are trying to {reason} but you've set only {nameof(IMinimalAdsMediator)}. Please, use {nameof(MagifyService.SetAdsMediator)} method with {nameof(IAdsMediator)} implementation.");
            return false;
        }

        internal BaseAdvertiserService(
            [NotNull] MagifySettings settings,
            [NotNull] ServicePrefs prefs,
            [CanBeNull] CampaignsProvider campaigns,
            [NotNull] AdPreloader adPreloader,
            [NotNull] INetworkStatusProvider network,
            bool isBannerCampaignRequired = false)
        {
            _settings = settings;
            _prefs = prefs;
            _campaigns = campaigns;
            AdPreloader = adPreloader;
            _network = network;
            IsBannerCampaignRequired = isBannerCampaignRequired;

            _network.Reachability
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => RefreshBanner())
                .AddTo(Disposables);

            BannerRequired
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => RefreshBanner())
                .AddTo(Disposables);

            _bannerDeactivationsCounter
                .SkipLatestValueOnSubscribe()
                .Subscribe(_ => RefreshBanner())
                .AddTo(Disposables);
        }

        void IDisposable.Dispose()
        {
            DisposeCurrentMediator();
            Disposables.Dispose();
        }

        /// <summary>
        /// Set the mediator for the service. It will be used to track impressions. <br/>
        /// <b>If you want to use ads preloading or public API, kind of: </b>
        /// <ul>
        /// <li> <see cref="BaseAdvertiserService.ShowInterVideoAsync"/> </li>
        /// <li> <see cref="BaseAdvertiserService.LoadInterVideoAsync"/> </li>
        /// <li> <see cref="BaseAdvertiserService.ShowRewardedVideoAsync"/> </li>
        /// <li> etc. </li>
        /// </ul>
        /// <b>so, you should pass <see cref="Magify.IAdsMediator"/> implementation here</b>
        /// </summary>
        public virtual void SetAdsMediator([CanBeNull] IMinimalAdsMediator mediator)
        {
            DisposeCurrentMediator();
            MinimalMediator = mediator;
            if (MinimalMediator != null)
            {
                MinimalMediator.OnBannerLoaded += BannerLoadedHandler;
                MinimalMediator.OnRewardShown += RewardShownHandler;
                MinimalMediator.OnInterShown += InterShownHandler;
            }
            if (mediator is IAdsMediator fullMediator)
            {
                RefreshBanner();
                AdPreloader.SetAdsMediator(fullMediator);
            }
        }

        private void DisposeCurrentMediator()
        {
            if (MinimalMediator != null)
            {
                MinimalMediator.OnBannerLoaded -= BannerLoadedHandler;
                MinimalMediator.OnRewardShown -= RewardShownHandler;
                MinimalMediator.OnInterShown -= InterShownHandler;
                if (IsBannerActive.Value && MinimalMediator is IAdsMediator { IsBannerVisible: true } mediator)
                {
                    mediator.HideBanner();
                }
                BannerActive.Value = BannerRequired.Value = false;
                MinimalMediator = null;
                AdPreloader.SetAdsMediator(null);
            }
        }

        #region Banner

        private void BannerLoadedHandler([NotNull] IAdsImpression impression)
        {
            MagifyManager.TrackAdsImpression(CampaignType.Banner, impression);
        }

        [NotNull]
        public IDisposable TemporaryDisableBanner()
        {
            return new DisposableCounter(_bannerDeactivationsCounter);
        }

        /// <summary>
        /// Refreshes current banner state
        /// </summary>
        public void RefreshBanner()
        {
            if (!CheckMediator("refresh banner") || !Mediator!.IsInitialized)
            {
                return;
            }

            // Check global state, current temporary deactivations and magify campaign availability to decide whether to show or hide the banner
            var needBanner = IsBannerRequired.Value && _bannerDeactivationsCounter.Value <= 0;
            if (needBanner && IsBannerCampaignRequired)
            {
                needBanner = _campaigns?.GetCampaignToHandle(_settings.BannerDefaultEvent, null, true) is BannerCampaign;
            }

            Logger.Log($"{nameof(RefreshBanner)} - " +
                       $"GloballyRequired={IsBannerRequired.Value}; " +
                       $"TemporaryDeactivations={_bannerDeactivationsCounter.Value}; " +
                       $"ShouldBeShown={needBanner}; " +
                       $"CurrentlyShown={BannerActive.Value};");

            // Do nothing if current banner state same as required
            if (needBanner == BannerActive.Value && needBanner == Mediator.IsBannerVisible)
            {
                Logger.Log($"{nameof(RefreshBanner)} - banner already {(needBanner ? "shown" : "hidden")}. ");
                return;
            }

            if (needBanner)
            {
                if (IsBannerCampaignRequired)
                {
                    _campaigns?.GetCampaignToHandle(_settings.BannerDefaultEvent, null);
                }
                if (!_network.IsNetworkReachable)
                {
                    Logger.Log($"{nameof(RefreshBanner)} - can't show because there is no internet");
                    return;
                }
                Logger.Log($"{nameof(RefreshBanner)} - asking {nameof(IAdsMediator)} to show banner");
                Mediator.ShowBanner();
            }
            else
            {
                Logger.Log($"{nameof(RefreshBanner)} - asking {nameof(IAdsMediator)} to hide banner");
                Mediator.HideBanner();
            }
            BannerActive.Value = needBanner;
        }

        #endregion

        #region Reward

        private void RewardShownHandler([NotNull] IAdsImpression impression)
        {
            MagifyManager.TrackAdsImpression(CampaignType.RewardedVideo, impression);
        }

        public UniTask<AdLoadResult> LoadRewardVideoAsync(CancellationToken cancellationToken)
        {
            var timeout = (float)_settings.RewardLoadTimeout;
            return LoadRewardVideoAsync(timeout, cancellationToken);
        }

        public async UniTask<AdLoadResult> LoadRewardVideoAsync(float timeout, CancellationToken cancellationToken)
        {
            if (!CheckMediator("load rewarded video"))
            {
                return AdLoadResult.NotReady();
            }
            if (timeout <= 0 && (!Mediator!.IsInitialized || !Mediator.IsRewardedVideoReady))
            {
                return AdLoadResult.Timeout();
            }
            if (Mediator!.IsInitialized && Mediator.IsRewardedVideoReady)
            {
                return AdLoadResult.Ready();
            }
            if (!_network.IsNetworkReachable)
            {
                Logger.LogWarning("Network is not reachable");
                return AdLoadResult.NoInternet();
            }

            return await LoadVideoAsync(LoadRewardVideoAsync, timeout, cancellationToken);
        }

        public async UniTask<AdShowResult> ShowRewardedVideoAsync([CanBeNull] string placement, [CanBeNull] Action<IAdsImpression> clickedCallback, CancellationToken cancellationToken)
        {
            if (!CheckMediator("show rewarded video"))
            {
                return AdShowResult.NotReady();
            }
            var ready = Mediator!.IsRewardedVideoReady;
            Logger.Log($"Show RewardVideo. {nameof(Mediator.IsRewardedVideoReady)} - {ready}");
            if (!ready)
            {
                return AdShowResult.NotReady();
            }

            var promise = new UniTaskCompletionSource<AdShowResult>();
            var rewardReceived = false;

            Mediator.OnRewardDisplayFailed += failed;
            Mediator.OnRewardHidden += hidden;
            Mediator.OnRewardClicked += clicked;
            Mediator.OnRewardReceived += rewarded;
            var result = AdShowResult.NotReady();
            try
            {
                Logger.Log($"Show RewardVideo for placement {placement}");
                _onBeforeShowRewardVideo.OnNext(Unit.Default);
                Mediator.ShowRewardVideo(placement);
                result = await promise.Task.AttachExternalCancellation(cancellationToken);
                Logger.Log($"Show RewardVideo finished with result {result}");
                return result;
            }
            finally
            {
                await UniTask.SwitchToMainThread(cancellationToken);
                if (result.State == AdShowState.Showed)
                {
                    _prefs.RewardedVideoCounter.Value++;
                }
                _onAfterShowRewardVideo.OnNext(Unit.Default);
                Mediator.OnRewardDisplayFailed -= failed;
                Mediator.OnRewardHidden -= hidden;
                Mediator.OnRewardClicked -= clicked;
                Mediator.OnRewardReceived -= rewarded;
            }

            void failed([CanBeNull] string error)
            {
                promise.TrySetResult(AdShowResult.MediatorError(error));
            }

            void rewarded()
            {
                Logger.Log("RewardVideo - reward received");
                rewardReceived = true;
            }

            void clicked([NotNull] IAdsImpression impression)
            {
                clickedCallback?.Invoke(impression);
            }

            void hidden([CanBeNull] IAdsImpression impression)
            {
                promise.TrySetResult(rewardReceived ? AdShowResult.Showed(impression) : AdShowResult.UserCanceled());
            }
        }

        private async UniTask<AdLoadResult> LoadRewardVideoAsync()
        {
            Logger.Log("RewardVideo - Ask SDK to load reward video");
            var mediator = Mediator;
            if (mediator!.IsRewardedVideoReady)
            {
                return AdLoadResult.Ready();
            }
            var promise = new UniTaskCompletionSource<AdLoadResult>();
            mediator.OnRewardLoaded += loaded;
            mediator.OnRewardLoadFailed += failed;

            mediator.LoadRewardVideo();
            try
            {
                return await promise.Task;
            }
            catch (Exception e)
            {
                return AdLoadResult.Unexpected(e.GetType()!.Name);
            }
            finally
            {
                await UniTask.SwitchToMainThread();
                mediator.OnRewardLoaded -= loaded;
                mediator.OnRewardLoadFailed -= failed;
            }

            void loaded()
            {
                promise.TrySetResult(AdLoadResult.Ready());
            }

            void failed([CanBeNull] string error)
            {
                promise.TrySetResult(AdLoadResult.MediatorError(error));
            }
        }

        #endregion

        #region Interstitial

        private void InterShownHandler([NotNull] IAdsImpression impression)
        {
            MagifyManager.TrackAdsImpression(CampaignType.Interstitial, impression);
        }

        public async UniTask<AdLoadResult> LoadInterVideoAsync(float timeout, CancellationToken cancellationToken)
        {
            if (!CheckMediator("load interstitial video"))
            {
                return AdLoadResult.NotReady();
            }
            if (timeout <= 0 && (!Mediator!.IsInitialized || !Mediator.IsInterVideoReady))
            {
                return AdLoadResult.Timeout();
            }
            if (Mediator!.IsInitialized && Mediator.IsRewardedVideoReady)
            {
                return AdLoadResult.Ready();
            }
            if (!_network.IsNetworkReachable)
            {
                Logger.LogWarning("Network is not reachable");
                return AdLoadResult.NoInternet();
            }
            return await LoadVideoAsync(LoadInterVideoAsync, timeout, cancellationToken);
        }

        public async UniTask<AdShowResult> ShowInterVideoAsync([CanBeNull] string placement, [CanBeNull] Action<IAdsImpression> clickedCallback)
        {
            if (!CheckMediator("show interstitial video"))
            {
                return AdShowResult.NotReady();
            }
            var mediator = Mediator;
            var ready = mediator!.IsInterVideoReady;
            Logger.Log($"Show InterVideo. {nameof(mediator.IsInterVideoReady)} - {ready}");

            var promise = new UniTaskCompletionSource<AdShowResult>();

            mediator.OnInterDisplayFailed += failed;
            mediator.OnInterHidden += hidden;
            mediator.OnInterClicked += clicked;
            var result = AdShowResult.NotReady();
            try
            {
                Logger.Log($"Show InterVideo");
                _onBeforeShowInterstitialVideo.OnNext(Unit.Default);
                mediator.ShowInterVideo(placement);
                result = await promise.Task;
                Logger.Log($"Show InterVideo finished with result {result}");
                return result;
            }
            finally
            {
                await UniTask.SwitchToMainThread();
                if (result.State == AdShowState.Showed)
                {
                    _prefs.InterstitialVideoCounter.Value++;
                }
                _onAfterShowInterstitialVideo.OnNext(Unit.Default);
                mediator.OnInterDisplayFailed -= failed;
                mediator.OnInterHidden -= hidden;
                mediator.OnInterClicked -= clicked;
            }

            void failed([CanBeNull] string error)
            {
                promise.TrySetResult(AdShowResult.MediatorError(error));
            }

            void hidden([NotNull] IAdsImpression impression)
            {
                promise.TrySetResult(AdShowResult.Showed(impression));
            }

            void clicked([NotNull] IAdsImpression impression)
            {
                clickedCallback?.Invoke(impression);
            }
        }

        private async UniTask<AdLoadResult> LoadInterVideoAsync()
        {
            Logger.Log("InterVideo - Ask SDK to load interstitial video");
            var mediator = Mediator;
            if (mediator!.IsInterVideoReady)
            {
                return AdLoadResult.Ready();
            }

            var promise = new UniTaskCompletionSource<AdLoadResult>();
            mediator.OnInterLoaded += loaded;
            mediator.OnInterLoadFailed += failed;

            mediator.LoadInterVideo();
            try
            {
                return await promise.Task;
            }
            catch (Exception e)
            {
                return AdLoadResult.Unexpected(e.GetType()!.Name);
            }
            finally
            {
                await UniTask.SwitchToMainThread();
                mediator.OnInterLoaded -= loaded;
                mediator.OnInterLoadFailed -= failed;
            }

            void loaded()
            {
                promise.TrySetResult(AdLoadResult.Ready());
            }

            void failed([CanBeNull] string error)
            {
                promise.TrySetResult(AdLoadResult.MediatorError(error));
            }
        }

        #endregion

        public async UniTask<bool> WaitForInitialization(float timeout, CancellationToken cancellationToken)
        {
            if (!CheckMediator("wait for initialization"))
            {
                return false;
            }
            var startTime = Time.time;
            while (Time.time - startTime < timeout)
            {
                if (!CheckMediator("wait for initialization"))
                {
                    Logger.LogWarning("Mediator was disposed during waiting for initialization, so we can't wait anymore, returning false");
                    return false;
                }
                if (Mediator!.IsInitialized)
                {
                    return true;
                }
                await UniTask.Yield(cancellationToken);
            }
            return false;
        }

        private async UniTask<AdLoadResult> LoadVideoAsync([NotNull] Func<UniTask<AdLoadResult>> repeater, float timeout, CancellationToken cancellationToken)
        {
            if (!Mediator!.IsInitialized)
            {
                var time = Time.time;
                if (!await WaitForInitialization(timeout, cancellationToken))
                {
                    return AdLoadResult.Timeout();
                }
                timeout -= Time.time - time;
            }

            AdLoadResult result;
            var source = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
            try
            {
                result = await repeater
                    .RepeatWhile(c => c?.State is AdLoadState.Loaded, source.Token)
                    .Timeout(TimeSpan.FromSeconds(timeout));
            }
            catch (TimeoutException)
            {
                result = AdLoadResult.Timeout();
                source.Cancel();
            }
            catch (Exception e)
            {
                result = AdLoadResult.Unexpected(e.GetType()!.Name);
            }
            finally
            {
                await UniTask.SwitchToMainThread();
                source.Dispose();
            }
            return result;
        }
    }
}