﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading.Tasks;
using UnityEditor.Compilation;
using UnityEditor.PackageManager;
using UnityEngine;

namespace Magify.DependenciesInstaller
{
    public enum InstallationMode
    {
        None,
        ByPackageName,
        ByGitUrl,
        ManualDownload,
    }

    [Serializable]
    public class ManualDownload
    {
        [field: SerializeField]
        public string Url { get; private set; }

        [field: SerializeField]
        public string NameWithExtension { get; private set; }
    }

    [Serializable]
    internal class DependencyElementConfiguration
    {
        [field: SerializeField]
        public string DisplayName { get; private set; }

        [field: SerializeField] [field: TextArea(0, 10)]
        public string Description { get; private set; }

        [field: Header("Checking:")]
        [field: SerializeField] [field: Space]
        public string PackageId { get; private set; }

        [field: SerializeField]
        public string AssemblyName { get; private set; }

        [field: Header("Installation:")]
        [field: SerializeField] [field: Header("By package name")]
        public string PackageNameId { get; private set; }

        [field: SerializeField]
        public ScopedRegistry Registry { get; private set; }

        [field: SerializeField] [field: Header("By git url")]
        public string GitUrlId { get; private set; }

        [field: SerializeField] [field: Header("Manual download")]
        public ManualDownload ManualDownload { get; private set; }

        [field: SerializeField] [field: Space]
        public string ReadMoreLink { get; private set; }

        public bool IsInstalled { get; set; }
        public InstallationMode InstallationMode { get; set; }

        public DependencyElementConfiguration Copy()
        {
            return new()
            {
                IsInstalled = IsInstalled,
                InstallationMode = InstallationMode,
                DisplayName = DisplayName,
                Description = Description,
                PackageId = PackageId,
                AssemblyName = AssemblyName,
                PackageNameId = PackageNameId,
                Registry = Registry,
                GitUrlId = GitUrlId,
                ManualDownload = ManualDownload,
                ReadMoreLink = ReadMoreLink,
            };
        }
    }

    internal class DependencyInstallerConfiguration : ScriptableObject
    {
        [field: SerializeField]
        public List<DependencyElementConfiguration> Dependencies { get; private set; }

        public async Task<IReadOnlyList<DependencyElementConfiguration>> GetElements()
        {
            var result = new List<DependencyElementConfiguration>(Dependencies.Count);
            var listRequest = Client.List(true);
            while (!listRequest.IsCompleted) await Task.Yield();
            var installed = listRequest.Result.Select(dep => dep.name).ToList();
            var assemblies = CompilationPipeline.GetAssemblies().Select(asm => asm.name).ToList();
            foreach (var dependencyConfig in Dependencies)
            {
                var element = dependencyConfig.Copy();
                element.InstallationMode = InstallationMode.None;
                element.IsInstalled = installed.Contains(dependencyConfig.PackageId) || assemblies.Contains(dependencyConfig.AssemblyName);
                result.Add(element);
            }

            return result;
        }
    }
}