using System;
using System.Threading;
using Cysharp.Threading.Tasks;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    internal static class ApplicationHelper
    {
        public enum RestoreType
        {
            Unsupported,
            ForegroundEvent,
            Timer,
        }

        private static IObservable<bool> _everyApplicationFocus;

        public static bool IsEditor(this RuntimePlatform platform)
        {
            return platform is RuntimePlatform.WindowsEditor or RuntimePlatform.LinuxEditor or RuntimePlatform.OSXEditor;
        }

        public static async UniTask<RestoreType> CallAndWaitForApplicationRestore(Action method, CancellationToken cancellationToken)
        {
            if (method == null)
            {
                throw new ArgumentNullException(nameof(method));
            }

            // This method well tested only for Android, iOS and Editor. Ignore awaiting focus on other platforms
            if (!Application.isEditor && Application.platform != RuntimePlatform.Android && Application.platform != RuntimePlatform.IPhonePlayer)
            {
                method();
                return RestoreType.Unsupported;
            }

            using var disposer = new CompositeDisposable();
            var source = new UniTaskCompletionSource<RestoreType>();

            EveryApplicationFocus()
                .Where(c => c && !source.IsCompleted())
                .ObserveOnMainThread(disposer.GetOrCreateToken())
                .Subscribe(_ => { source.TrySetResult(RestoreType.ForegroundEvent); })
                .AddTo(disposer);

            // In some cases system does not send focus and pause events correctly.
            // For that matter finish awaiting after half second
            if (!Application.isEditor)
            {
                Observable.Timer(TimeSpan.FromSeconds(0.5))
                    .Where(_ => !source.IsCompleted())
                    .Subscribe(_ => { source.TrySetResult(RestoreType.Timer); })
                    .AddTo(disposer);
            }

            method();

            return await source.Task.AttachExternalCancellation(cancellationToken);
        }

        /// <summary>
        /// Same as Observable.EveryApplicationFocus(), but is not invoking when focus is lost on the NotificationCenter shown.
        /// Use ObserveOnMainThread() if you want to handle it on main thread
        /// </summary>
        public static IObservable<bool> EveryApplicationFocus()
        {
            switch (Application.platform)
            {
                case RuntimePlatform.Android or RuntimePlatform.IPhonePlayer when !Application.isEditor && MagifyManager.Initialized:
                    return _everyApplicationFocus ??= Observable.FromEvent<bool>(t => MagifyManager.OnApplicationFocus += t, t => MagifyManager.OnApplicationFocus -= t);
                default:
                    return Observable.EveryApplicationFocus();
            }
        }
    }
}