using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class FeaturesProvider : IDisposable
    {
        public delegate bool TryGetFunc<T>(string featureName, out T result);
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        [NotNull]
        private readonly MagifySettings _settings;
        private readonly Dictionary<string, ReactiveProperty<bool>> _booleans = new();
        private readonly Dictionary<string, ReactiveProperty<double>> _numbers = new();
        private readonly Dictionary<string, ReactiveProperty<string>> _strings = new();

        public bool LogIfFeatureIsNotRemote { get; set; } = true;
        /// <summary>Only <b>LogType.Log</b> | <b>LogType.Warning</b> | <b>LogType.Error</b> are possible</summary>
        public LogType IfFeatureIsNotRemoteLogType { get; set; } = LogType.Warning;

        /// <inheritdoc cref="MagifyManager.Features.OnFeaturesParsed"/>
        public event Action<ConfigKind> OnFeaturesParsed
        {
            add => MagifyManager.Features.OnFeaturesParsed += value;
            remove => MagifyManager.Features.OnFeaturesParsed -= value;
        }

        /// <inheritdoc cref="MagifyManager.Features.OnFeaturesUpdated"/>
        public event Action OnFeaturesUpdated
        {
            add => MagifyManager.Features.OnFeaturesUpdated += value;
            remove => MagifyManager.Features.OnFeaturesUpdated -= value;
        }

        [NotNull]
        public IStoredAppFeaturesCollection StoredAppFeatures => MagifyManager.Features.StoredAppFeatures;

        [NotNull]
        public NumberRemoteProperty StoredAppFeaturesLoadingTimeoutSeconds => _settings.StoredAppFeaturesLoadingTimeoutSeconds;

        public FeaturesProvider([NotNull] MagifySettings settings)
        {
            _settings = settings;
            MagifyManager.Features.OnFeaturesUpdated += FeaturesUpdatedHandler;
        }

        public FeaturesProvider()
        {
            MagifyManager.Features.OnFeaturesUpdated += FeaturesUpdatedHandler;
        }

        public bool TryGetBool(string featureName, out Feature<bool> result) => MagifyManager.Features.TryGetBool(featureName, out result) && VerifyFeature(result, featureName) is not null;
        /// <exception cref="Magify.MagifyFeatureNotFoundException">When a feature by a given name was not found</exception>
        public Feature<bool> GetBool(string featureName) => VerifyFeature(MagifyManager.Features.GetBool(featureName), featureName);
        public IReadOnlyReactiveProperty<bool> GetReactiveBool(string featureName, bool defaultValue = default)
        {
            _booleans.TryAdd(featureName, new ReactiveProperty<bool>(defaultValue));
            return _booleans[featureName];
        }

        public bool TryGetNumber(string featureName, out Feature<double> result) => MagifyManager.Features.TryGetNumber(featureName, out result) && VerifyFeature(result, featureName) is not null;
        /// <exception cref="Magify.MagifyFeatureNotFoundException">When a feature by a given name was not found</exception>
        public Feature<double> GetNumber(string featureName) => VerifyFeature(MagifyManager.Features.GetNumber(featureName), featureName);
        public IReadOnlyReactiveProperty<double> GetReactiveNumber(string featureName, double defaultValue = default)
        {
            _numbers.TryAdd(featureName, new ReactiveProperty<double>(defaultValue));
            return _numbers[featureName];
        }

        public bool TryGetString(string featureName, out Feature<string> result) => MagifyManager.Features.TryGetString(featureName, out result) && VerifyFeature(result, featureName) is not null;
        /// <exception cref="Magify.MagifyFeatureNotFoundException">When a feature by a given name was not found</exception>
        public Feature<string> GetString(string featureName) => VerifyFeature(MagifyManager.Features.GetString(featureName), featureName);
        public IReadOnlyReactiveProperty<string> GetReactiveString(string featureName, string defaultValue = default)
        {
            _strings.TryAdd(featureName, new ReactiveProperty<string>(defaultValue));
            return _strings[featureName];
        }

        /// <inheritdoc cref="MagifyManager.Features.SetIgnoredFeatures"/>
        public void SetIgnoredFeatures([CanBeNull] IEnumerable<string> featuresToIgnore, FeatureSource source) => MagifyManager.Features.SetIgnoredFeatures(featuresToIgnore, source);

        private Feature<T> VerifyFeature<T>(Feature<T> feature, string name)
        {
            if (LogIfFeatureIsNotRemote && feature.IsDefault)
            {
                _logger.Log($"Feature '{name}' value ({feature.Value}) is not remote. You got the feature from the default config (the remote config has not been loaded or doesn't contain this feature).", IfFeatureIsNotRemoteLogType);
            }
            return feature;
        }

        private void FeaturesUpdatedHandler()
        {
            _logger.Log("Update reactive features");
            // Here is no need to call VerifyFeature() because after MagifyService initialization FeaturesUpdatedHandler might be called only for
            updateValues(_booleans, MagifyManager.Features.TryGetBool);
            updateValues(_numbers, MagifyManager.Features.TryGetNumber);
            updateValues(_strings, MagifyManager.Features.TryGetString);

            void updateValues<T>(Dictionary<string, ReactiveProperty<T>> map, TryGetFunc<Feature<T>> getter)
            {
                foreach (var (featureName, property) in map)
                {
                    if (getter(featureName, out var feature))
                    {
                        property.Value = feature.Value;
                    }
                }
            }
        }

        void IDisposable.Dispose()
        {
            MagifyManager.Features.OnFeaturesUpdated -= FeaturesUpdatedHandler;
        }

        /// <summary>
        /// Requests to download the feature content.
        /// Firstly a web request is sent to check if we already have the actual content value on disk.
        /// If there is, it will be loaded from disk, otherwise the content will be downloaded and saved for later use.
        /// </summary>
        /// <returns>
        /// Content Handle: The content will remain in the cache as long as there is at least one handle that has not been disposed of.
        /// Remember to dispose of it when you want to release content from RAM if you don't want a memory leak.
        /// </returns>
        [NotNull, ItemCanBeNull]
        public async UniTask<ContentHandle<StoredAppFeatureContent>> LoadStoredAppFeature(StoredAppFeature feature, CancellationToken cancellationToken, double? timeoutSeconds = null)
        {
            timeoutSeconds ??= StoredAppFeaturesLoadingTimeoutSeconds.Value;
            var handle = await MagifyManager.Storage.LoadStoredAppFeature(feature.Url, timeoutSeconds.Value, cancellationToken);
            return handle?.Value is null ? default : handle;
        }

        /// <inheritdoc cref="LoadStoredAppFeature(Magify.StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>
        [NotNull, ItemCanBeNull]
        public async UniTask<ContentHandle<StoredAppFeatureContent>> LoadStoredAppFeature([NotNull] string featureName, CancellationToken cancellationToken, double? timeoutSeconds = null)
        {
            if (StoredAppFeatures.TryGet(featureName, out var feature))
            {
                return await LoadStoredAppFeature(feature, cancellationToken, timeoutSeconds);
            }
            _logger.LogWarning($"Stored app feature with name {featureName} does not exists");
            return default;
        }

        /// <summary>
        /// The same as <see cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>, but for many features: <br/>
        /// <inheritdoc cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>
        /// </summary>
        [NotNull, ItemNotNull]
        public async UniTask<IEnumerable<ContentHandle<StoredAppFeatureContent>>> LoadStoredAppFeatures([NotNull] IEnumerable<StoredAppFeature> features, CancellationToken cancellationToken, double? timeoutSeconds = null)
        {
            timeoutSeconds ??= StoredAppFeaturesLoadingTimeoutSeconds.Value;
            var loaded = await UniTask.WhenAll(features.Select(loader));
            return loaded == null || loaded.Length == 0
                ? ArraySegment<ContentHandle<StoredAppFeatureContent>>.Empty
                : loaded.Where(notNull);

            UniTask<ContentHandle<StoredAppFeatureContent>> loader(StoredAppFeature feature) => MagifyManager.Storage.LoadStoredAppFeature(feature.Url, timeoutSeconds.Value, cancellationToken);
            bool notNull([CanBeNull] ContentHandle<StoredAppFeatureContent> handle) => handle?.Value != null;
        }

        /// <summary>
        /// The same as <see cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>, but for many features: <br/>
        /// <inheritdoc cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>
        /// </summary>
        [NotNull, ItemNotNull]
        public async UniTask<IEnumerable<ContentHandle<StoredAppFeatureContent>>> LoadStoredAppFeatures([NotNull, ItemNotNull] IEnumerable<string> features, CancellationToken cancellationToken, double? timeoutSeconds = null)
        {
            timeoutSeconds ??= StoredAppFeaturesLoadingTimeoutSeconds.Value;
            var loaded = await UniTask.WhenAll(features.Select(loader));
            return loaded == null || loaded.Length == 0
                ? ArraySegment<ContentHandle<StoredAppFeatureContent>>.Empty
                : loaded.Where(notNull);

            bool notNull([CanBeNull] ContentHandle<StoredAppFeatureContent> handle) => handle?.Value != null;
            UniTask<ContentHandle<StoredAppFeatureContent>> loader([NotNull] string featureName)
            {
                if (StoredAppFeatures.TryGet(featureName, out var feature))
                    return MagifyManager.Storage.LoadStoredAppFeature(feature.Url, timeoutSeconds.Value, cancellationToken);
                _logger.LogWarning($"Stored app feature with name {featureName} does not exists");
                return UniTask.FromResult<ContentHandle<StoredAppFeatureContent>>(null);
            }
        }

        /// <summary>
        /// The same as <see cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>, but for all features: <br/>
        /// <inheritdoc cref="LoadStoredAppFeature(StoredAppFeature,System.Threading.CancellationToken,System.Nullable{double})"/>
        /// </summary>
        [NotNull, ItemNotNull]
        public async UniTask<IEnumerable<ContentHandle<StoredAppFeatureContent>>> LoadStoredAppFeatures(CancellationToken cancellationToken, double? timeoutSeconds = null)
        {
            timeoutSeconds ??= StoredAppFeaturesLoadingTimeoutSeconds.Value;
            var loaded = await UniTask.WhenAll(StoredAppFeatures.Select(loader));
            return loaded == null || loaded.Length == 0
                ? ArraySegment<ContentHandle<StoredAppFeatureContent>>.Empty
                : loaded.Where(notNull);

            UniTask<ContentHandle<StoredAppFeatureContent>> loader(StoredAppFeature feature) => MagifyManager.Storage.LoadStoredAppFeature(feature.Url, timeoutSeconds.Value, cancellationToken);
            bool notNull([CanBeNull] ContentHandle<StoredAppFeatureContent> handle) => handle?.Value != null;
        }
    }
}