using System;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public class AdvertiserService : BaseAdvertiserService, IDisposable
    {
        public override IMinimalAdsMediator MinimalMediator => Mediator;
        public override IAdsMediator Mediator { get; protected set; }

        internal AdvertiserService(
            [NotNull] MagifySettings settings,
            [NotNull] ServicePrefs prefs,
            [NotNull] CampaignsProvider campaigns,
            [NotNull] ProductsObtainer products,
            [NotNull] AdPreloader adPreloader,
            [NotNull] INetworkStatusProvider network) : base(settings, prefs, campaigns, adPreloader, network, isBannerCampaignRequired: true)
        {
            products.OnProductObtained
                .Subscribe(_ => RefreshBanner())
                .AddTo(Disposables);
        }

        void IDisposable.Dispose()
        {
            DisposeCurrentMediator();
            Disposables.Dispose();
        }

        /// <summary> <b>
        /// You should use SetAdsMediator(IAdsMediator mediator) in full version of Magify SDK
        /// </b> </summary>
        public override void SetAdsMediator([CanBeNull] IMinimalAdsMediator mediator)
        {
            Logger.LogWarning("You should use SetAdsMediator(IAdsMediator mediator) in full version of Magify SDK");
            SetAdsMediator(mediator as IAdsMediator);
        }

        /// <summary>
        /// Sets the mediator for the showing and tracking ads.
        /// </summary>
        public void SetAdsMediator([CanBeNull] IAdsMediator mediator)
        {
            DisposeCurrentMediator();
            Mediator = mediator;
            if (Mediator != null)
            {
                Mediator.OnBannerLoaded += BannerLoadedHandler;
                Mediator.OnBannerLoadFailed += BannerLoadFailedHandler;
                Mediator.OnBannerClicked += BannerClickedHandler;
                RefreshBanner();
            }
            AdPreloader.SetAdsMediator(mediator);
        }

        private void DisposeCurrentMediator()
        {
            if (Mediator != null)
            {
                if (IsBannerActive.Value && Mediator.IsBannerVisible)
                {
                    Mediator.HideBanner();
                }
                BannerActive.Value = BannerRequired.Value = false;
                Mediator.OnBannerLoaded -= BannerLoadedHandler;
                Mediator.OnBannerLoadFailed -= BannerLoadFailedHandler;
                Mediator.OnBannerClicked -= BannerClickedHandler;
            }
            AdPreloader.SetAdsMediator(null);
        }

        protected virtual void BannerLoadedHandler([NotNull] IAdsImpression impression)
        {
#if MAGIFY_3_OR_NEWER
            var campaignName = _campaigns.GetCampaign(_settings.BannerDefaultEvent)?.Name;
            if (campaignName == null)
            {
                _logger.LogWarning($"The banner was loaded, but there was no campaign on the default banner event ({_settings.BannerDefaultEvent})");
            }
            TrackAdsImpression(campaignName ?? string.Empty, impression);
#else
            MagifyManager.TrackAdsImpression(CampaignType.Banner, impression);
#endif
        }
        private void BannerLoadFailedHandler([CanBeNull] string reason)
        {
#if MAGIFY_3_OR_NEWER
            var campaignName = _campaigns.GetCampaign(_settings.BannerDefaultEvent)?.Name;
            if (campaignName == null)
            {
                _logger.LogWarning($"The banner was failed to show, but there was no campaign on the default banner event ({_settings.BannerDefaultEvent})");
            }
            TrackImpressionFailFor(campaignName ?? string.Empty, reason);
#else
            MagifyManager.TrackImpressionFailFor(CampaignType.Banner, reason);
#endif
        }

        private void BannerClickedHandler([CanBeNull] IAdsImpression obj)
        {
#if MAGIFY_3_OR_NEWER
            var campaignName = _campaigns.GetCampaign(_settings.BannerDefaultEvent)?.Name;
            if (campaignName == null)
            {
                _logger.LogWarning($"The banner was clicked, but there was no campaign on the default banner event ({_settings.BannerDefaultEvent})");
            }
            TrackAdsClickFor(campaignName ?? string.Empty);
#else
            MagifyManager.TrackAdsClickFor(CampaignType.Banner);
#endif
        }
    }
}