﻿#if UNITY_PURCHASES
using System;
using Newtonsoft.Json.Linq;
using UnityEngine;
using UnityEngine.Purchasing;

namespace Magify
{
    public static class UnityPurchasesExt
    {
        private const string OrderIdKey = "orderId";
        private const string PurchaseTokenKey = "purchaseToken";

        /// <summary>
        /// Gets the Order ID (Android) or Transaction ID (other platforms) from a Product.
        /// </summary>
        /// <param name="product">The Product for which to get the Order ID or Transaction ID.</param>
        /// <returns>The Order ID or Transaction ID, or an empty string if an exception occurs.</returns>
        public static string GetTransactionID(this Product product)
        {
            return Application.platform switch
            {
                RuntimePlatform.Android => GetOrderIdFromReceipt(product.receipt),
                _ => product.transactionID,
            };
        }

        /// <summary>
        /// Gets the Order ID (Android) or Original Transaction ID (iOS) or Transaction ID (other platforms) from a Product.
        /// </summary>
        /// <param name="product">The Product for which to get the Order ID or Original Transaction ID or Transaction ID.</param>
        /// <returns>The Order ID or Original Transaction ID or Transaction ID, or an empty string if an exception occurs.</returns>
        public static string GetOriginalTransactionID(this Product product)
        {
            switch (Application.platform)
            {
                case RuntimePlatform.Android:
                    return GetOrderIdFromReceipt(product.receipt);
                case RuntimePlatform.IPhonePlayer:
                    return product.appleOriginalTransactionID;
                default:
                    return product.transactionID;
            }
        }

        /// <summary>
        /// Parses a Product receipt to get the Order ID (Android).
        /// </summary>
        /// <param name="jsonString">The JSON string of the Product receipt.</param>
        /// <returns>The Order ID or an empty string if an exception occurs.</returns>
        public static string GetOrderIdFromReceipt(string jsonString)
        {
            return GetValueFromReceipt(jsonString, OrderIdKey);
        }

        /// <summary>
        /// Gets the Purchase Token (Android) or null (other platforms) from a Product.
        /// </summary>
        /// <param name="product">The Product for which to get the Purchase Token.</param>
        public static string GetPurchaseToken(this Product product)
        {
            return GetValueFromReceipt(product.receipt, PurchaseTokenKey);
        }

        public static string GetValueFromReceipt(string jsonString, string key)
        {
            if (Application.platform != RuntimePlatform.Android && !Application.platform.IsEditor())
            {
                return string.Empty;
            }
            try
            {
                var json = JObject.Parse(jsonString);
                var payloadJson = json["Payload"]!.ToString();
                var payload = JObject.Parse(payloadJson);
                var innerJson = payload["json"]!.ToString();
                var innerPayload = JObject.Parse(innerJson);
                return innerPayload[key]!.ToString();
            }
            catch (Exception)
            {
                return string.Empty;
            }
        }

        public static ProductObtainFailReason ToPurchaseFailReason(this PurchaseFailureReason desc)
        {
            return desc switch
            {
                PurchaseFailureReason.PurchasingUnavailable => ProductObtainFailReason.PurchasingUnavailable,
                PurchaseFailureReason.ExistingPurchasePending => ProductObtainFailReason.ExistingPurchasePending,
                PurchaseFailureReason.ProductUnavailable => ProductObtainFailReason.ProductUnavailable,
                PurchaseFailureReason.SignatureInvalid => ProductObtainFailReason.SignatureInvalid,
                PurchaseFailureReason.UserCancelled => ProductObtainFailReason.UserCancelled,
                PurchaseFailureReason.PaymentDeclined => ProductObtainFailReason.PaymentDeclined,
                PurchaseFailureReason.DuplicateTransaction => ProductObtainFailReason.DuplicateTransaction,
                _ => ProductObtainFailReason.Unknown,
            };
        }

        public static ProductObtainFailReason ToPurchaseFailReason(this InitializationFailureReason desc)
        {
            return desc switch
            {
                InitializationFailureReason.PurchasingUnavailable => ProductObtainFailReason.PurchasingUnavailable,
                InitializationFailureReason.NoProductsAvailable => ProductObtainFailReason.NoProductsAvailable,
                InitializationFailureReason.AppNotKnown => ProductObtainFailReason.AppNotKnown,
                _ => ProductObtainFailReason.Unknown,
            };
        }

        public static UnityEngine.Purchasing.ProductType ToProductType(this ProductDef product)
        {
            return product switch
            {
                InAppProduct { IsConsumable: true } => UnityEngine.Purchasing.ProductType.Consumable,
                InAppProduct => UnityEngine.Purchasing.ProductType.NonConsumable,
                SubscriptionProduct => UnityEngine.Purchasing.ProductType.Subscription,
                _ => throw new ArgumentOutOfRangeException(nameof(product))
            };
        }
    }
}
#endif