#if UNITY_PURCHASES
using System;
using System.Collections.Generic;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using Unity.Services.Core;
using UnityEngine;
using UnityEngine.Purchasing;
using UnityEngine.Purchasing.Extension;
using JetBrains.Annotations;

namespace Magify
{
    public partial class UnityPurchases : IDetailedStoreListener, IInAppStore
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        private readonly INetworkStatusProvider _network;

        private UniTaskCompletionSource<ProductObtainFailReason?> _initializationPromise;
        private ProductObtainFailReason? _lastStoreError = null;
        private IStoreController _controller;
        private IExtensionProvider _extension;
        private ProductsFetcher _fetcher;

        private bool _purchasingIsBusy;
        private uint _operationIdCounter;

        public event Action<IStoreController, IExtensionProvider> OnInitialized;
        public event ProductFetchedDelegate OnProductFetched;
        public event ProductFetchFailedDelegate OnProductFetchFailed;
        public event PurchaseFinishedDelegate OnPurchaseFinished;
        public event PurchaseFinishedDelegate OnSubscriptionPurchaseFinished;
        public event PurchaseFailedDelegate OnPurchaseFailed;

        public bool IsInitialized => Controller != null;

        public PricesHelper Prices { get; }

        [CanBeNull]
        public IStoreController Controller => _controller;

        public IExtensionProvider Extension => _extension;

        public UnityPurchases(INetworkStatusProvider network, ILocalizer localizer)
        {
            _network = network;
            Prices = new PricesHelper(localizer);
        }

        #region Public API to work with InAps

        /// <summary>
        /// Does Purchases have information about <see cref="product"/> in local storage
        /// </summary>
        public bool IsProductReady(ProductDef product)
        {
            if (!product.IsPurchasable()) return true;
            if (!IsInitialized) return false;
            return Controller?.products.WithID(product.Id) != null;
        }

        /// <summary>
        /// Initiate fetching of <see cref="products"/>. The result will be call of <see cref="OnProductFetchFailed"/> in case of error or <see cref="OnProductFetched"/> if succeeded
        /// </summary>
        public void LoadProducts(IEnumerable<ProductDef> products)
        {
            LoadProductsAsync(products).Forget();
        }

        private async UniTaskVoid LoadProductsAsync(IEnumerable<ProductDef> products)
        {
            if (products is not IReadOnlyList<ProductDef> typed)
            {
                typed = products.ToArray();
            }
            try
            {
                var fetchResult = await FetchProductsAsync(typed);
                fetchFinished(fetchResult);
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                fetchFinished(ProductObtainFailReason.Unknown);
            }

            void fetchFinished(ProductObtainFailReason? failReason)
            {
                var reason = failReason ?? ProductObtainFailReason.Unknown;
                foreach (var product in typed)
                {
                    if (HasProduct(product.Id))
                    {
                        OnProductFetched?.Invoke(product.Id);
                    }
                    else
                    {
                        OnProductFetchFailed?.Invoke(product.Id, reason);
                    }
                }
            }
        }

        /// <summary>
        /// Initiate purchase of <see cref="product"/>. The result will be call of <see cref="OnPurchaseFailed"/> in case of error or <see cref="OnPurchaseFinished"/> if succeeded
        /// </summary>
        public void Purchase(ProductDef product)
        {
            PurchaseAsync(product).Forget();
        }

        private async UniTaskVoid PurchaseAsync(ProductDef product)
        {
            try
            {
                var obtainResult = await InitiatePurchaseAsync(product);
                if (obtainResult != null)
                {
                    PurchaseFailed(product.Id, obtainResult.Value);
                }
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                PurchaseFailed(product.Id, ProductObtainFailReason.Unknown);
            }
        }

        /// <remarks>
        /// Use it if you are SURE, that Unity Purchases already Initialized. It is important to remember, that some Stores have a specific logic on app minimize-maximize. Usually it's better to use this method only on application start.
        /// </remarks>
        public SubscriptionInfo LoadSubscriptionInfo(string productId)
        {
            return UnityPurchasesUtils.LoadSubscriptionInfo(productId, Controller, Extension);
        }

        /// <summary>
        /// Does Purchases have information about <see cref="product"/> in local storage
        /// </summary>
        public bool HasProduct(string product)
        {
            if (!IsInitialized) return false;
            return Controller?.products.WithID(product) != null;
        }

        /// <summary>
        /// Does Purchases have information about <see cref="products"/> in local storage
        /// </summary>
        public bool HasProducts(params string[] products)
        {
            return products.All(HasProduct);
        }

        /// <summary>
        /// Does Purchases have information about <see cref="products"/> in local storage
        /// </summary>
        public bool HasProducts(params ProductDef[] products)
        {
            return products.All(IsProductReady);
        }

        /// <summary>
        /// Does Purchases have information about <see cref="products"/> in local storage
        /// </summary>
        public bool HasProducts(IEnumerable<string> products)
        {
            return products.All(HasProduct);
        }

        /// <summary>
        /// Does Purchases have information about <see cref="products"/> in local storage
        /// </summary>
        public bool HasProducts(IEnumerable<ProductDef> products)
        {
            return products.All(IsProductReady);
        }

        /// <summary>
        /// Returns local information about <see cref="product"/>
        /// </summary>
        public UniTask<Product> GetProductAsync(ProductDef product)
        {
            return GetProductAsync(product.Id);
        }

        /// <summary>
        /// Returns local information about <see cref="product"/>
        /// </summary>
        public async UniTask<Product> GetProductAsync(string product, CancellationToken cancellationToken = default)
        {
            while (!IsInitialized)
            {
                await UniTask.Yield(cancellationToken);
            }

            return GetProduct(product);
        }

        /// <summary>
        /// Returns local information about <see cref="product"/>
        /// </summary>
        public Product GetProduct(ProductDef product)
        {
            return GetProduct(product.Id);
        }

        /// <summary>
        /// Returns local information about <see cref="product"/>
        /// </summary>
        public Product GetProduct(string product)
        {
            _logger.Log($"Getting local information about {product}.");
            if (!IsInitialized)
            {
                _logger.LogError("Store not ready to do that");
                return null;
            }
            var info = Controller.products.WithID(product);
            if (info == null)
            {
                _logger.LogError($"There is no information about {product}");
            }
            return info;
        }

        private async UniTask<ProductObtainFailReason?> FetchProductsAsync(IReadOnlyCollection<ProductDef> products)
        {
            var operationId = _operationIdCounter++;
            _logger.Log($"[FETCH {operationId}] Getting products information. Count: {products.Count}; {string.Join(", ", products.Select(c => c.Id))}");
            if (products.Any(c => !c.IsPurchasable()))
            {
                _logger.Log($"[FETCH {operationId}] Filter non-purchasable products: {string.Join(", ", products.Where(c => !c.IsPurchasable()).Select(c => c.Id))}");
                products = products.Where(c => c.IsPurchasable()).ToArray();
                _logger.Log($"[FETCH {operationId}] Remaining products in the collection: {string.Join(", ", products.Select(c => c.Id))}");
            }
            if (products.Count == 0)
            {
                _logger.Log($"[FETCH {operationId}] There is no products left to fetch");
                return ProductObtainFailReason.NoProductsAvailable;
            }
            if (products.All(c => HasProduct(c.Id)))
            {
                _logger.Log($"[FETCH {operationId}] All products already fetched");
                return null;
            }
            if (!_network.IsNetworkReachable)
            {
                _logger.Log($"[FETCH {operationId}] There is no internet - can't fetch new products");
                return ProductObtainFailReason.NoInternet;
            }
            if (_purchasingIsBusy)
            {
                _logger.LogError($"[FETCH {operationId}] It is not possible to fetch product information when restoring or purchasing in progress");
                return ProductObtainFailReason.PurchasingIsBusy;
            }

            _logger.Log($"[FETCH {operationId}] Switch to main thread before work with {nameof(UnityPurchasing)}");
            await UniTask.SwitchToMainThread();
            if (UnityServices.State is not ServicesInitializationState.Initialized)
            {
                _logger.Log($"[FETCH {operationId}] Initialize {nameof(UnityServices)} before use {nameof(UnityPurchasing)}");
                await UnityServices.InitializeAsync();
            }

            if (!IsInitialized)
            {
                _logger.Log($"[FETCH {operationId}] {nameof(UnityPurchasing)} isn't ready yet - try to initialize it");
                Func<UniTask> repeater = () => LoadPurchasingService(products);
                try
                {
                    await repeater.RepeatWhile(() => IsInitialized, CancellationToken.None, new ExponentialBackoff());
                    if (_lastStoreError != null)
                    {
                        _logger.LogError($"Failed to initialize store. Reason: {_lastStoreError.Value}");
                        return _lastStoreError.Value;
                    }
                }
                catch (Exception e)
                {
                    Debug.LogException(e);
                    _logger.LogWarning($"[FETCH {operationId}] Products fetching failed with exception. See console logs for more information");
                    return ProductObtainFailReason.Unknown;
                }
            }

            if (Application.isEditor)
            {
                await UniTask.Delay(1000);
            }

            _logger.Log($"[FETCH {operationId}] Call {nameof(ProductsFetcher)}.{nameof(ProductsFetcher.FetchProductsAsync)} and wait for result");
            var result = await _fetcher.FetchProductsAsync(operationId, products);
            _logger.Log($"[FETCH {operationId}] Call {nameof(ProductsFetcher)}.{nameof(ProductsFetcher.FetchProductsAsync)} finished with result: {result}");
            if (result != null && products.Any(c => !HasProduct(c.Id)))
            {
                _logger.LogError($"[FETCH {operationId}] Information about some products not found after fetch: {string.Join(", ", products.Select(c => c.Id).Where(c => !HasProduct(c)))}");
                result = ProductObtainFailReason.NoProductsAvailable;
            }
            foreach (var product in products.Where(IsProductReady).Select(GetProduct))
            {
                Prices.UpdatePriceFor(product.definition.id, product.metadata.localizedPriceString);
            }
            return result;
        }

        private async UniTask<ProductObtainFailReason?> InitiatePurchaseAsync(ProductDef productDef)
        {
            var operationId = _operationIdCounter++;
            _logger.Log($"[PURCHASE {operationId}] Begin purchasing of product: {productDef}");

            if (!IsInitialized)
            {
                _logger.LogError($"Can't purchase {productDef.Id} - purchasing not ready yet");
                return ProductObtainFailReason.PurchasingNotReady;
            }
            var product = Controller.products.WithID(productDef.Id);
            if (product == null)
            {
                _logger.LogError($"Can't purchase {productDef.Id} - product information not fetched");
                return ProductObtainFailReason.NoProductsAvailable;
            }
            if (_purchasingIsBusy)
            {
                _logger.LogError($"[PURCHASE {operationId}] Failed to buy product - purchasing is busy");
                return ProductObtainFailReason.PurchasingIsBusy;
            }
            if (!_network.IsNetworkReachable)
            {
                _logger.Log($"[PURCHASE {operationId}] Can't process purchase because there is no internet");
                return ProductObtainFailReason.NoInternet;
            }

            if (product.definition.type == UnityEngine.Purchasing.ProductType.Subscription && product.hasReceipt && productDef is SubscriptionProduct)
            {
                if (!IsInitialized) await UniTask.WaitUntil(() => IsInitialized);
                var unitySubscriptionModel = UnityPurchasesUtils.GetUnitySubscriptionModel(productDef.Id, Controller, Extension);
                if (unitySubscriptionModel != null && unitySubscriptionModel.IsExpired() == Result.False)
                {
                    _logger.Log($"[PURCHASE {operationId}] The product is {ProductType.Subscription} and it already bought and is not expired. Don't need to buy it again");
                    return ProductObtainFailReason.SubscriptionAlreadyBought;
                }
            }

            _logger.Log($"[PURCHASE {operationId}] Initiate purchase of product {productDef.Id} using {nameof(IStoreController)}.{nameof(IStoreController.InitiatePurchase)}");
            _purchasingIsBusy = true;

            if (Application.isEditor)
            {
                await UniTask.DelayFrame(5);
            }

            Controller.InitiatePurchase(productDef.Id);

            return null;
        }

        // public async UniTask<RestorePurchasesResult> RestorePurchases()
        // {
        //     var operationId = _operationIdCounter++;
        //     _logger.Log($"[RESTORE {operationId}] Restore purchases requested");
        //     if (Application.platform != RuntimePlatform.IPhonePlayer)
        //     {
        //         _logger.Log($"[RESTORE {operationId}] Platform {Application.platform} does not support purchases restoring");
        //         return RestorePurchasesResult.NothingRestore;
        //     }
        //
        //     if (!_network.IsNetworkReachable)
        //     {
        //         _logger.LogWarning("[RESTORE {operationId}] Can't restore purchases because there is no internet");
        //         return RestorePurchasesResult.NoInternet;
        //     }
        //
        //     if (_initializationPromise != null)
        //     {
        //         _logger.Log($"[RESTORE {operationId}] There is initialization of {nameof(UnityPurchasing)} in progress - waiting");
        //         await _initializationPromise.Task;
        //         _logger.Log($"[RESTORE {operationId}] Initialization of {nameof(UnityPurchasing)} finished with result '{_lastStoreError}'. Continue purchasing restoring if possible");
        //     }
        //
        //     if (!IsReady)
        //     {
        //         _logger.LogWarning($"[RESTORE {operationId}] Can't restore purchases because store is not ready. Result of last try is '{_lastStoreError}'");
        //         return RestorePurchasesResult.StoreNotReady;
        //     }
        //
        //     if (_purchasingIsBusy)
        //     {
        //         _logger.LogError($"[RESTORE {operationId}] It is not possible to restore products - another restoring or purchasing in progress");
        //         return RestorePurchasesResult.PurchasingIsBusy;
        //     }
        //
        //     _logger.Log($"[RESTORE {operationId}] Getting {nameof(IAppleExtensions)} from {nameof(UnityPurchasing)}");
        //     var appleStore = Extension.GetExtension<IAppleExtensions>();
        //     if (appleStore == null)
        //     {
        //         _logger.LogError($"[RESTORE {operationId}] Can't restore purchases because {nameof(IAppleExtensions)} not found in {nameof(UnityPurchasing)}");
        //         return RestorePurchasesResult.StoreError;
        //     }
        //
        //     var anyRestore = false;
        //     _purchasingIsBusy = true;
        //
        //     try
        //     {
        //         OnPurchaseFinished += onPurchaseRestored;
        //         _logger.Log($"[RESTORE {operationId}] Call {nameof(IAppleExtensions)}.{nameof(IAppleExtensions.RestoreTransactions)} and wait for result.");
        //         using (await _spinner.ShowAsDisposable())
        //         {
        //             var restorePromise = new UniTaskCompletionSource<(bool Successful, string Error)>();
        //             appleStore.RestoreTransactions((successful, error) => { restorePromise.TrySetResult((successful, error)); });
        //             var result = await restorePromise.Task;
        //             if (result.Successful)
        //             {
        //                 _logger.Log($"[RESTORE {operationId}] {nameof(IAppleExtensions)}.{nameof(IAppleExtensions.RestoreTransactions)} finished successfully");
        //                 _logger.Log($"[RESTORE {operationId}] Give store {_settings.PurchasesRestoreTimeout} seconds for restoration callbacks");
        //                 await UniTask.Delay(TimeSpan.FromSeconds(_settings.PurchasesRestoreTimeout));
        //                 if (anyRestore)
        //                 {
        //                     _logger.Log($"[RESTORE {operationId}] Some purchases were restored");
        //                     return RestorePurchasesResult.RestoreSuccessful;
        //                 }
        //                 else
        //                 {
        //                     _logger.Log($"[RESTORE {operationId}] No product has been restored.");
        //                     return RestorePurchasesResult.NothingRestore;
        //                 }
        //             }
        //             else
        //             {
        //                 _logger.Log($"[RESTORE {operationId}] {nameof(IAppleExtensions)}.{nameof(IAppleExtensions.RestoreTransactions)} failed. Message: {result.Error}");
        //                 await UniTask.Delay(TimeSpan.FromSeconds(.5f));
        //                 return RestorePurchasesResult.RestoreFail;
        //             }
        //         }
        //     }
        //     finally
        //     {
        //         _purchasingIsBusy = false;
        //         OnPurchaseFinished -= onPurchaseRestored;
        //     }
        //
        //     void onPurchaseRestored(string productId, PurchaseInfo purchaseInfo)
        //     {
        //         _logger.Log($"[RESTORE {operationId}] Purchase of product {productId} was restored");
        //         anyRestore = true;
        //         MagifyManager.TrackRestoredInAppFor(productId);
        //     }
        // }

        #endregion

        #region Internal functionality

        private async UniTask LoadPurchasingService(IReadOnlyCollection<ProductDef> items)
        {
            if (_initializationPromise != null)
            {
                _logger.Log("There is another initialization in process. Just wait for it...");
                await _initializationPromise.Task;
                return;
            }

            _initializationPromise = new UniTaskCompletionSource<ProductObtainFailReason?>();
            _lastStoreError = null;
            try
            {
                _logger.Log($"Prepare products catalog and add all products ({items.Count}): {string.Join(", ", items.Select(c => c.Id))}");
                var builder = ConfigurationBuilder.Instance(StandardPurchasingModule.Instance());
                foreach (var def in items)
                {
                    var productType = def.ToProductType();
                    var payouts = def.Payout.Select(payout => new PayoutDefinition(payout.Type, payout.Quantity));
                    builder.AddProduct(def.Id, productType, null, payouts);
                }

                _logger.Log("Initialize unity purchasing");
                UnityPurchasing.Initialize(this, builder);
            }
            catch (Exception e)
            {
                Debug.LogException(e);
                _initializationPromise.TrySetResult(ProductObtainFailReason.Unknown);
            }

            _logger.Log("Waiting for internal initialization");
            _lastStoreError = await _initializationPromise.Task;
            _logger.Log($"Internal initialization finished. Result: {_lastStoreError}");
            _initializationPromise = null;
        }

        #endregion

        #region Implementation of IStoreListener

        /// <summary>
        /// Called when Unity IAP is ready to make purchases.
        /// </summary>
        void IStoreListener.OnInitialized(IStoreController controller, IExtensionProvider extensions)
        {
            _logger.Log("Store initialization succeeded");
            _extension = extensions;
            _controller = controller;
            _fetcher = new ProductsFetcher(Controller);
            _initializationPromise.TrySetResult(null);
            OnInitialized?.Invoke(controller, extensions);
        }

        /// <summary>
        /// Purchasing failed to initialise for a non recoverable reason.
        /// </summary>
        /// <param name="error"> The failure reason. </param>
        void IStoreListener.OnInitializeFailed(InitializationFailureReason error)
        {
            OnInitializeFailed(error, "");
        }

        /// <summary>
        /// Purchasing failed to initialise for a non recoverable reason.
        /// </summary>
        /// <param name="error"> The failure reason. </param>
        /// <param name="message"> More detail on the error : for example the GoogleBillingResponseCode. </param>
        public void OnInitializeFailed(InitializationFailureReason error, string message)
        {
            _logger.LogError($"Store initialization failed with reason {error}");
            _extension = null;
            _controller = null;
            _fetcher = null;
            _initializationPromise.TrySetResult(error.ToPurchaseFailReason());
        }

        /// <summary>
        /// Called when a purchase completes. May be called at any time after OnInitialized().
        /// </summary>
        PurchaseProcessingResult IStoreListener.ProcessPurchase(PurchaseEventArgs e)
        {
            if (e is null)
            {
                _logger.LogError($"Unity's sent null {nameof(PurchaseEventArgs)} to the {nameof(IStoreListener)} implementation of type {nameof(UnityPurchases)}. It's impossible and cannot be handled.");
                return PurchaseProcessingResult.Pending; // it's better to try to handle it on the next application start
            }

            _logger.Log($"IStoreListener: Succeed to purchase {e.purchasedProduct.definition.id}");
            _purchasingIsBusy = false;
            return ProcessPurchaseInternal(e);
        }

        protected virtual PurchaseProcessingResult ProcessPurchaseInternal([NotNull] PurchaseEventArgs e)
        {
            var product = e.purchasedProduct;
            var purchaseInfo = new PurchaseInfo(product, LoadSubscriptionInfo(product.definition.id));
            if (purchaseInfo.SubscriptionInfo != null) OnSubscriptionPurchaseFinished?.Invoke(product.definition.id, purchaseInfo);
            OnPurchaseFinished?.Invoke(product.definition.id, purchaseInfo);
            return PurchaseProcessingResult.Complete;
        }

        /// <summary>
        /// Called when a purchase fails.
        /// </summary>
        void IStoreListener.OnPurchaseFailed(Product product, PurchaseFailureReason reason)
        {
            _logger.Log($"IStoreListener: Failed to purchase {product.definition.id}. Reason: {reason}");
            _purchasingIsBusy = false;
            PurchaseFailed(product.definition.id, reason.ToPurchaseFailReason());
        }

        /// <summary>
        /// Called when a purchase fails.
        /// </summary>
        void IDetailedStoreListener.OnPurchaseFailed(Product product, PurchaseFailureDescription desc)
        {
            _logger.Log($"IDetailedStoreListener: Failed to purchase {product.definition.id}. Reason: {desc.reason}. Message: {desc.message}");
            _purchasingIsBusy = false;
            PurchaseFailed(product.definition.id, desc.reason.ToPurchaseFailReason());
        }

        protected void PurchaseFailed(string productId, ProductObtainFailReason reason)
        {
            OnPurchaseFailed?.Invoke(productId, reason);
        }

        protected void SubscriptionPurchaseFinished(string productId, PurchaseInfo purchaseInfo)
        {
            OnSubscriptionPurchaseFinished?.Invoke(productId, purchaseInfo);
        }

        protected void PurchaseFinished(string productId, PurchaseInfo purchaseInfo)
        {
            OnPurchaseFinished?.Invoke(productId, purchaseInfo);
        }

        #endregion
    }
}

#endif