﻿using System;
using Newtonsoft.Json;
using UnityEngine;
using UnityEngine.Scripting;

namespace Magify
{
    [Preserve]
    public class PurchaseReceiptModel
    {
        private static readonly MagifyLogger _magifyLogger = MagifyLogger.Get();

        /// <summary>
        /// The name of the store in use, such as GooglePlay or AppleAppStore
        /// </summary>
        [JsonProperty("Store"), Preserve]
        public string Store { get; init; }

        /// <summary>
        /// This transaction’s unique identifier, provided by the store
        /// </summary>
        [JsonProperty("TransactionID"), Preserve]
        public string TransactionID { get; init; }

        /// <summary>
        /// Varies by platform: <br/>
        ///
        /// For Apple App Store - Base64 encoded receipt
        /// (<a href="https://docs.unity3d.com/Packages/com.unity.purchasing@4.5/manual/AppleReceipt.html">detailed for unity purchasing</a>) <br/>
        ///
        /// For Google Play - JSON hash with the following keys and values: json and signature
        /// (<a href="https://docs.unity3d.com/Packages/com.unity.purchasing@4.5/manual/GoogleReceipt.html">detailed for unity purchasing</a>)
        ///
        /// <example>
        /// <b>How to get raw value from Apple App Store:</b> <br/>
        /// var <b>builder</b> = UnityEngine.Purchasing.ConfigurationBuilder.Instance(UnityEngine.Purchasing.StandardPurchasingModule.Instance()); <br/>
        /// var <b>receipt</b> = builder.Configure&lt;UnityEngine.Purchasing.IAppleConfiguration&gt;().appReceipt;
        /// </example>
        /// </summary>
        [JsonProperty("Payload"), Preserve]
        public string Payload { get; init; }

        public PurchaseReceiptModel(string json)
        {
            if (string.IsNullOrWhiteSpace(json))
            {
                _magifyLogger.Log("Receipt json is empty, empty receipt will be created.");
                return;
            }
            var model = JsonFacade.DeserializeObject<PurchaseReceiptModel>(json)!;
            Store = model.Store;
            TransactionID = model.TransactionID;
            Payload = model.Payload;
        }

        public override string ToString()
        {
            try
            {
                return JsonFacade.SerializeObject(this);
            }
            catch (Exception e)
            {
                _magifyLogger.LogException(e);
                return "";
            }
        }
    }
}