using System;
using System.Collections.Generic;
using JetBrains.Annotations;
using Magify.Rx;

namespace Magify
{
    public class MixedUnityAghanimStore : IInAppStore, ICancelable
    {
        public event PurchaseFinishedDelegate OnPurchaseFinished;
        public event ProductFetchedDelegate OnProductFetched;
        public event ProductFetchFailedDelegate OnProductFetchFailed;
        public event PurchaseFailedDelegate OnPurchaseFailed;

        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();

        [NotNull]
        public ICollection<IDisposable> Disposables => _disposables;
        [NotNull]
        public IInAppStore UnityStore { get; private set; }
        [NotNull]
        public IInAppStore AghanimStore { get; private set; }

        public bool IsDisposed => _disposables.IsDisposed;

        public MixedUnityAghanimStore([NotNull] IInAppStore unityStore, [NotNull] IInAppStore aghanimStore)
        {
            if (unityStore is IDisposable unityDisposable)
                _disposables.Add(unityDisposable);
            if (aghanimStore is IDisposable aghanimDisposable)
                _disposables.Add(aghanimDisposable);
            UnityStore = unityStore;
            AghanimStore = aghanimStore;

            UnityStore.OnPurchaseFinished += HandlePurchaseFinished;
            UnityStore.OnPurchaseFailed += HandlePurchaseFailed;
            UnityStore.OnProductFetched += HandleProductFetched;
            UnityStore.OnProductFetchFailed += HandleProductFetchFailed;

            AghanimStore.OnPurchaseFinished += HandlePurchaseFinished;
            AghanimStore.OnPurchaseFailed += HandlePurchaseFailed;
            AghanimStore.OnProductFetched += HandleProductFetched;
            AghanimStore.OnProductFetchFailed += HandleProductFetchFailed;
        }

        public bool IsProductReady(ProductDef product)
        {
            if (IsSuitableForAghanim(product))
            {
                return AghanimStore.IsProductReady(product);
            }
            return UnityStore.IsProductReady(product);
        }

        public void LoadProducts(IEnumerable<ProductDef> products)
        {
            if (products == null)
            {
                return;
            }
            var toLoadForUnity = new List<ProductDef>(16);
            var toLoadForAghanim = new List<ProductDef>(16);
            foreach (var product in products)
            {
                if (IsSuitableForAghanim(product))
                {
                    toLoadForAghanim.Add(product);
                }
                else
                {
                    toLoadForUnity.Add(product);
                }
            }
            UnityStore.LoadProducts(toLoadForUnity);
            AghanimStore.LoadProducts(toLoadForAghanim);
        }

        public void Purchase(ProductDef product)
        {
            if (IsSuitableForAghanim(product))
            {
                AghanimStore.Purchase(product);
            }
            else
            {
                UnityStore.Purchase(product);
            }
        }

        public SubscriptionInfo LoadSubscriptionInfo(string productId)
        {
            return UnityStore.LoadSubscriptionInfo(productId);
        }

        private static bool IsSuitableForAghanim(ProductDef product)
        {
            return product is InAppProduct inAppProduct && inAppProduct.Store == PurchaseStore.Aghanim;
        }

        private void HandleProductFetchFailed(string productId, ProductObtainFailReason reason) => OnProductFetchFailed?.Invoke(productId, reason);
        private void HandleProductFetched(string productId) => OnProductFetched?.Invoke(productId);
        private void HandlePurchaseFailed(string productId, ProductObtainFailReason reason) => OnPurchaseFailed?.Invoke(productId, reason);
        private void HandlePurchaseFinished(string productId, PurchaseInfo purchaseInfo) => OnPurchaseFinished?.Invoke(productId, purchaseInfo);
        public void Dispose()
        {
            _disposables.Release();
        }
    }
}