﻿using System;
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Threading;
using Cysharp.Threading.Tasks;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    /// <inheritdoc cref="MagifyDocs.NoAghanimSubsFunc"/>
    public class AghanimStore : IInAppStore, ICancelable
    {
        [NotNull]
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);
        [NotNull]
        private readonly AghanimStorePrefs _prefs;
        [NotNull]
        private readonly PooledCompositeDisposable _disposables = new();
        [NotNull]
        private readonly HashSet<string> _processingOrders = new();

        [NotNull]
        protected IExternalLinkHandler PurchaseLinkHandler { get; }

        public bool IsSandboxMode { get; set; } = false;

        public bool IsDisposed => _disposables.IsDisposed;

        public event ProductFetchedDelegate OnProductFetched;
        public event ProductFetchFailedDelegate OnProductFetchFailed;
        public event PurchaseFailedDelegate OnPurchaseFailed;
        public event PurchaseFinishedDelegate OnPurchaseFinished;

        public AghanimStore([NotNull] IExternalLinkHandler purchaseLinkHandler)
        {
            _prefs = new AghanimStorePrefs(Path.Combine(PackageInfo.PersistentPath, "aghanim-prefs.bin"))
                .AddTo(_disposables);
            PurchaseLinkHandler = purchaseLinkHandler;
            MagifyManager.Aghanim.OnOrderStatusChanged
                .Where(pair => !_processingOrders.Contains(pair.OrderId))
                .Subscribe(HandleExternalOrderStatusChanged)
                .AddTo(_disposables);
        }

        public void Dispose()
        {
            _disposables.Release();
        }

        public bool IsProductReady(ProductDef product)
        {
            return product is InAppProduct inAppProduct && inAppProduct.Store == PurchaseStore.Aghanim;
        }

        public async UniTask RestorePurchases(CancellationToken cancellationToken)
        {
            cancellationToken = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, _disposables.GetOrCreateToken()).Token;
            var purchasedProducts = await MagifyManager.Aghanim.LoadPurchasedProductsAsync(cancellationToken);
            if (purchasedProducts == null || purchasedProducts.Count == 0)
            {
                return;
            }
            foreach (var (productId, count) in purchasedProducts)
            {
                if (productId == null)
                {
                    continue;
                }
                var product = MagifyManager.InAppProducts.FirstOrDefault(p => p != null && p.Id == productId && p.Store == PurchaseStore.Aghanim);
                if (product is { IsConsumable: false })
                {
                    var unprocessed = _prefs.ProcessedPurchasedProducts.TryGetValue(productId, out var processedCount)
                        ? count - processedCount
                        : count;
                    for (var i = 0; i < unprocessed; i++)
                    {
                        OnPurchaseFinished?.Invoke(productId, PurchaseInfo.ForRestored(PurchaseStore.Aghanim));
                    }
                }
            }
        }

        public void LoadProducts(IEnumerable<ProductDef> products)
        {
            if (products == null)
                return;

            foreach (var product in products)
            {
                if (product is InAppProduct && IsProductReady(product))
                {
                    OnProductFetched?.Invoke(product.Id);
                }
                else if (product is not null)
                {
                    OnProductFetchFailed?.Invoke(product.Id, ProductObtainFailReason.ProductUnavailable);
                }
            }

            foreach (var orderInfo in MagifyManager.Aghanim.IterateAllOrders())
            {
                var orderId = orderInfo.Id;
                if (_prefs.ProcessedOrders.Contains(orderId))
                    continue;
                if (!MagifyManager.Aghanim.TryGetProductOfOrder(orderId, out var productId))
                    continue;
                TryHandleOrder(productId, orderId, orderInfo);
            }
        }

        public void Purchase(ProductDef product)
        {
            if (product != null)
            {
                PurchaseAsync(product, _disposables.GetOrCreateToken()).Forget();
            }
        }

        public void Purchase(ProductDef product, CancellationToken cancellationToken)
        {
            if (product != null)
            {
                cancellationToken = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken, _disposables.GetOrCreateToken()).Token;
                PurchaseAsync(product, cancellationToken).Forget();
            }
        }

        protected async UniTaskVoid PurchaseAsync([NotNull] ProductDef product, CancellationToken cancellation)
        {
            AghanimOrder order = default;
            try
            {
                order = await MagifyManager.Aghanim.IssueOrder(product.Id, IsSandboxMode, cancellation);
                if (order == null || string.IsNullOrWhiteSpace(order.Url))
                {
                    PurchaseFailed(product.Id, ProductObtainFailReason.ProductUnavailable);
                    return;
                }
                _processingOrders.Add(order.Id);
                await TaskScheduler.SwitchToMainThread(cancellation);
                await PurchaseLinkHandler.Open(order.Url, cancellation);
                var orderStatus = await MagifyManager.Aghanim.WaitForFinalOrderStatusAsync(order.Id, cancellation);
                TryHandleOrder(product.Id, order.Id, new AghanimOrderInfo(order.Id, orderStatus, order.Price, order.Currency));
            }
            catch (Exception e)
            {
                _logger.LogException(e);
                PurchaseFailed(product.Id, ProductObtainFailReason.Unknown);
            }
            finally
            {
                if (order != null)
                {
                    _processingOrders.Remove(order.Id);
                }
            }
        }

        protected bool TryHandleOrder([NotNull] string productId, [NotNull] string orderId, AghanimOrderInfo orderInfo)
        {
            if (_prefs.ProcessedOrders.Contains(orderId))
            {
                return false;
            }
            if (orderInfo.Status is AghanimOrderStatus.Paid)
            {
                var info = new PurchaseInfo(orderInfo.Price, orderInfo.Currency, orderId, orderId, purchaseToken: null, receipt: null, PurchaseStore.Aghanim, skipVerification: true);
                if (PurchaseFinished(productId, info))
                    _prefs.ProcessedOrders.Add(orderId);
                return true;
            }
            if (orderInfo.Status is AghanimOrderStatus.Cancelled)
            {
                if (PurchaseFailed(productId, ProductObtainFailReason.UserCancelled))
                    _prefs.ProcessedOrders.Add(orderId);
                return true;
            }
            return false;
        }

        private void HandleExternalOrderStatusChanged((string OrderId, AghanimOrderStatus Status) tuple)
        {
            if (tuple.OrderId != null)
            {
                HandleExternalOrderStatusChanged(tuple.OrderId, tuple.Status);
            }
        }

        protected virtual void HandleExternalOrderStatusChanged([NotNull] string orderId, AghanimOrderStatus status)
        {
            if (MagifyManager.Aghanim.TryGetProductOfOrder(orderId, out var productId)
             && MagifyManager.Aghanim.TryGetOrderInfo(orderId, out var info))
            {
                TryHandleOrder(productId, orderId, info);
            }
        }

        /// <inheritdoc cref="MagifyDocs.NoAghanimSubsFunc"/>
        /// <returns>
        /// Always null
        /// </returns>
        [Obsolete(MagifyDocs.NoAghanimSubsFunc), CanBeNull]
        public SubscriptionInfo LoadSubscriptionInfo(string productId)
        {
            return null;
        }

        protected bool PurchaseFailed(string productId, ProductObtainFailReason reason)
        {
            try
            {
                OnPurchaseFailed?.Invoke(productId, reason);
                return true;
            }
            catch (Exception e)
            {
                _logger.LogException(e);
            }
            return false;
        }

        protected bool PurchaseFinished(string productId, PurchaseInfo purchaseInfo)
        {
            try
            {
                OnPurchaseFinished?.Invoke(productId, purchaseInfo);
                return true;
            }
            catch (Exception e)
            {
                _logger.LogException(e);
            }
            return false;
        }
    }
}