using System.Collections.Generic;
using System.IO;
using System.Threading;
using Cysharp.Threading.Tasks;
using UnityEngine;

namespace Magify
{
    public class ResourcesPopupsProvider : IPopupsProvider
    {
        private const string RateReviewPopupId = "RateReviewPopup";
        private const string InterPopupSplashId = "InterPopupSplash";
        private const string ErrorPopupId = "ErrorPopup";
        private const string SpinnerId = "Spinner";
        private const string ImagePopupId = "ImagePopup";
        private const string FullscreenPopupId = "FullscreenPopup";

        private readonly string _path;
        private readonly Dictionary<object, GameObject> _loadedPopups = new();

        public ResourcesPopupsProvider(string path)
        {
            _path = path;
        }

        protected virtual string GetCreativeResourcePath(string screenId)
        {
            return screenId;
        }

        public void UnloadPopup<TArgs>(IPopupBase<TArgs> panel)
            where TArgs : PopupArgs
        {
            if (_loadedPopups.TryGetValue(panel, out var gameObject))
            {
                _loadedPopups.Remove(panel);
                if (Application.isPlaying)
                {
                    Object.Destroy(gameObject);
                }
            }
        }

        UniTask<IPopupWithProducts<CreativePopup.Args>> IPopupsProvider.LoadCreativeWithProductsPopup(string screenId, CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopupWithProducts<CreativePopup.Args>, CreativePopup.Args>(GetCreativeResourcePath(screenId), cancellationToken);
        }

        UniTask<IPopup<PopupArgs>> IPopupsProvider.LoadPopup(string screenId, CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopup<PopupArgs>, PopupArgs>(GetCreativeResourcePath(screenId), cancellationToken);
        }

        UniTask<IPopup<ImagePopup.Args>> IPopupsProvider.LoadImagePopup(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopup<ImagePopup.Args>, ImagePopup.Args>(ImagePopupId, cancellationToken);
        }

        UniTask<IPopup<ImagePopup.Args>> IPopupsProvider.LoadFullscreenPopup(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopup<ImagePopup.Args>, ImagePopup.Args>(FullscreenPopupId, cancellationToken);
        }

        UniTask<IPopupBase<PopupArgs>> IPopupsProvider.LoadSpinner(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopupBase<PopupArgs>, PopupArgs>(SpinnerId, cancellationToken);
        }

        UniTask<IPopup<PopupArgs>> IPopupsProvider.LoadRateReviewPopup(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopup<PopupArgs>, PopupArgs>(RateReviewPopupId, cancellationToken);
        }

        UniTask<IPopupBase<PopupArgs>> IPopupsProvider.LoadInterPopupSplash(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopupBase<PopupArgs>, PopupArgs>(InterPopupSplashId, cancellationToken);
        }

        UniTask<IPopup<ErrorPopup.Args>> IPopupsProvider.LoadErrorPopup(CancellationToken cancellationToken)
        {
            return LoadPopupFromResources<IPopup<ErrorPopup.Args>, ErrorPopup.Args>(ErrorPopupId, cancellationToken);
        }

        private async UniTask<TPopup> LoadPopupFromResources<TPopup, TArgs>(string screenId, CancellationToken cancellationToken)
            where TPopup : class, IPopupBase<TArgs>
            where TArgs : PopupArgs
        {
            var prefabPath = Path.Combine(_path, screenId);
            var gameObject = await Resources.LoadAsync(prefabPath).ToUniTask(cancellationToken: cancellationToken) as GameObject;
            if (gameObject == null)
            {
                return default;
            }
            gameObject = Object.Instantiate(gameObject, MagifyService.Instance.Presenter.Canvas.transform);
            MagifyService.Instance.Presenter.RecalculateSafeArea();
            var panel = gameObject.GetComponent<TPopup>();
            if (panel == null)
            {
                Object.Destroy(gameObject);
                return null;
            }
            _loadedPopups[panel] = gameObject;
            return panel;
        }
    }
}