#if LEVELPLAY_MEDIATOR
using System;
using Magify.Rx;
using UnityEngine;

namespace Magify
{
    public class LevelPlayMediator : IAdsMediator
    {
        private static readonly MagifyLogger _logger = MagifyLogger.Get(MagifyService.LogScope);

        #region BannerEvents

        public event Action<string> OnBannerLoadFailed;
        public event Action<IAdsImpression> OnBannerLoaded;
        public event Action<IAdsImpression> OnBannerClicked;

        #endregion

        #region Reward Events

        public event Action<IAdsImpression> OnRewardShown;
        public event Action OnRewardLoaded;
        public event Action<string> OnRewardLoadFailed;
        public event Action<string> OnRewardDisplayFailed;
        public event Action OnRewardReceived;
        public event Action<IAdsImpression> OnRewardClicked;
        public event Action<IAdsImpression> OnRewardHidden;

        #endregion

        #region Inter Events

        public event Action<IAdsImpression> OnInterShown;
        public event Action OnInterLoaded;
        public event Action<string> OnInterLoadFailed;
        public event Action<string> OnInterDisplayFailed;
        public event Action<IAdsImpression> OnInterClicked;
        public event Action<IAdsImpression> OnInterHidden;

        #endregion

        private readonly MagifySettings _settings;
        private bool _isInitialized;
        private IronSourceAdInfo _rewardInfo;
        private IronSourceAdInfo _interInfo;
        private CompositeDisposable _disposables;

        public bool IsInitialized => _isInitialized;
        public bool IsBannerVisible { get; private set; }
        public bool IsRewardedVideoReady => _rewardInfo != null && IronSource.Agent.isRewardedVideoAvailable();
        public bool IsInterVideoReady => _interInfo != null && IronSource.Agent.isInterstitialReady();

        public string ApkKey { get; private set; }

        public LevelPlayMediator(MagifySettings settings)
        {
            _settings = settings;
        }

        public void Initialize(bool testSuite = false)
        {
            _disposables = new CompositeDisposable();

            switch (Application.platform)
            {
                case RuntimePlatform.Android:
                    ApkKey = _settings.ApkKeyGP;
                    break;
                case RuntimePlatform.IPhonePlayer:
                    ApkKey = _settings.ApkKeyIOS;
                    break;
                case RuntimePlatform.LinuxEditor:
                case RuntimePlatform.WindowsEditor:
                case RuntimePlatform.OSXEditor:
                    ApkKey = _settings.ApkKeyGP;
                    break;
            }

            IronSource.Agent.setUserId(MagifyManager.ClientId);
            _logger.Log("Check IronSource.Agent.validateIntegration");
            IronSource.Agent.validateIntegration();
            IronSource.Agent.setManualLoadRewardedVideo(true);
            if (testSuite)
            {
                _logger.Log("Set IronSource \"is_test_suite\" flag enabled");
                IronSource.Agent.setMetaData("is_test_suite", "enable");
            }

            Observable.EveryApplicationPause()
                .Subscribe(pause => IronSource.Agent.onApplicationPause(pause))
                .AddTo(_disposables);

            IronSourceEvents.onSdkInitializationCompletedEvent += OnSdkInitialized;

            IronSourceBannerEvents.onAdClickedEvent += BannerAdClickedHandler;
            IronSourceBannerEvents.onAdLoadedEvent += BannerAdLoadedHandler;
            IronSourceBannerEvents.onAdLoadFailedEvent += BannerAdFailedHandler;

            IronSourceRewardedVideoEvents.onAdClosedEvent += RaiseRewardClosed;
            IronSourceRewardedVideoEvents.onAdReadyEvent += RaiseRewardLoaded;
            IronSourceRewardedVideoEvents.onAdLoadFailedEvent += RaiseRewardLoadFailed;
            IronSourceRewardedVideoEvents.onAdShowFailedEvent += RaiseRewardDisplayFailed;
            IronSourceRewardedVideoEvents.onAdRewardedEvent += RaiseRewardReceived;
            IronSourceRewardedVideoEvents.onAdClosedEvent += RaiseRewardHidden;
            IronSourceRewardedVideoEvents.onAdClickedEvent += RaiseRewardClicked;

            IronSourceInterstitialEvents.onAdClosedEvent += RaiseInterClosed;
            IronSourceInterstitialEvents.onAdReadyEvent += RaiseInterLoaded;
            IronSourceInterstitialEvents.onAdLoadFailedEvent += RaiseInterLoadFailed;
            IronSourceInterstitialEvents.onAdShowFailedEvent += RaiseInterDisplayFailed;
            IronSourceInterstitialEvents.onAdClickedEvent += RaiseInterClicked;
            IronSourceInterstitialEvents.onAdClosedEvent += RaiseInterHidden;

            _logger.Log("Call IronSource.Agent.init");
            IronSource.Agent.init(ApkKey);
        }

        private void OnSdkInitialized()
        {
            _logger.Log("OnSdkInitialized success");
            _isInitialized = true;
            IronSourceEvents.onSdkInitializationCompletedEvent -= OnSdkInitialized;
        }

        public void Terminate()
        {
            _logger.Log("Terminate LevelPlay");
            _disposables?.Dispose();

            _interInfo = null;
            _rewardInfo = null;

            IronSourceBannerEvents.onAdClickedEvent -= BannerAdClickedHandler;
            IronSourceBannerEvents.onAdLoadedEvent -= BannerAdLoadedHandler;
            IronSourceBannerEvents.onAdLoadFailedEvent -= BannerAdFailedHandler;

            IronSourceRewardedVideoEvents.onAdReadyEvent -= RaiseRewardLoaded;
            IronSourceRewardedVideoEvents.onAdLoadFailedEvent -= RaiseRewardLoadFailed;
            IronSourceRewardedVideoEvents.onAdShowFailedEvent -= RaiseRewardDisplayFailed;
            IronSourceRewardedVideoEvents.onAdRewardedEvent -= RaiseRewardReceived;
            IronSourceRewardedVideoEvents.onAdClickedEvent -= RaiseRewardClicked;
            IronSourceRewardedVideoEvents.onAdClosedEvent -= RaiseRewardHidden;

            IronSourceInterstitialEvents.onAdReadyEvent -= RaiseInterLoaded;
            IronSourceInterstitialEvents.onAdLoadFailedEvent -= RaiseInterLoadFailed;
            IronSourceInterstitialEvents.onAdShowFailedEvent -= RaiseInterDisplayFailed;
            IronSourceInterstitialEvents.onAdClickedEvent -= RaiseInterClicked;
            IronSourceInterstitialEvents.onAdClosedEvent -= RaiseInterHidden;
        }

        #region Banner

        public void ShowBanner()
        {
            _logger.Log($"{nameof(ShowBanner)} - create, setup and show banner");
            IronSource.Agent.loadBanner(IronSourceBannerSize.BANNER, IronSourceBannerPosition.BOTTOM);
        }

        public void HideBanner()
        {
            _logger.Log($"{nameof(ShowBanner)} - hide and destroy banner with");
            IronSource.Agent.destroyBanner();
            IsBannerVisible = false;
        }

        private void BannerAdClickedHandler(IronSourceAdInfo info)
        {
            _logger.Log($"Banner clicked");
            OnBannerClicked?.Invoke(GetImpression(info));
        }

        private void BannerAdLoadedHandler(IronSourceAdInfo info)
        {
            _logger.Log($"Banner expanded");
            IsBannerVisible = true;
            OnBannerLoaded?.Invoke(GetImpression(info));
        }

        private void BannerAdFailedHandler(IronSourceError info)
        {
            _logger.LogError($"Banner failed to display:{System.Environment.NewLine}{info.getDescription()}");
            IsBannerVisible = false;
            OnBannerLoadFailed?.Invoke(info.getDescription());
        }

        #endregion

        public void LoadRewardVideo()
        {
            _rewardInfo = null;
            IronSource.Agent.loadRewardedVideo();
        }

        public void LoadInterVideo()
        {
            _interInfo = null;
            IronSource.Agent.loadInterstitial();
        }

        public void ShowRewardVideo(string placement)
        {
            IronSource.Agent.showRewardedVideo(placement);
        }

        public void ShowInterVideo(string placement)
        {
            IronSource.Agent.showInterstitial(placement);
        }

        public void LaunchTestSuite()
        {
            IronSource.Agent.launchTestSuite();
        }

        private IAdsImpression GetImpression(IronSourceAdInfo info)
        {
            return new IronSourceAdsImpression
            {
                Id = info.instanceId,
                Network = info.adNetwork,
                AdUnit = info.adUnit,
                InstanceName = info.instanceName,
                Revenue = info.revenue,
                Precision = info.precision,
                Ab = info.ab,
                AuctionId = info.auctionId,
                SegmentName = info.segmentName,
                LifetimeRevenue = info.lifetimeRevenue,
                EncryptedCpm = info.encryptedCPM
            };
        }

        #region Raise reward events

        protected virtual void RaiseRewardClosed(IronSourceAdInfo info)
        {
            _logger.Log($"RewardVideo with id {info.auctionId} closed");
            OnRewardShown?.Invoke(GetImpression(info));
        }

        private void RaiseRewardLoaded(IronSourceAdInfo info)
        {
            _logger.Log($"Reward video - {info.auctionId} loaded. NetworkName: {info.adNetwork}; AdUnitIdentifier: {info.adUnit}; InstanceId: {info.instanceId}");
            _rewardInfo = info;
            OnRewardLoaded?.Invoke();
        }

        private void RaiseRewardLoadFailed(IronSourceError error)
        {
            _logger.LogError($"RewardVideo - Can't load reward video. Error: {error}");
            OnRewardLoadFailed?.Invoke(error.ToString());
        }

        private void RaiseRewardDisplayFailed(IronSourceError error, IronSourceAdInfo info)
        {
            _logger.LogError($"Failed to show RewardVideo with id {info.auctionId}");
            OnRewardDisplayFailed?.Invoke(error.ToString());
        }

        private void RaiseRewardReceived(IronSourcePlacement placement, IronSourceAdInfo info)
        {
            _logger.Log($"RewardVideo with id {info.auctionId} rewarded");
            OnRewardReceived?.Invoke();
        }

        private void RaiseRewardClicked(IronSourcePlacement placement, IronSourceAdInfo info)
        {
            _logger.Log($"RewardVideo with id {info.auctionId} clicked");
            OnRewardClicked?.Invoke(GetImpression(info));
        }

        private void RaiseRewardHidden(IronSourceAdInfo info)
        {
            _logger.Log($"RewardVideo with id {info.auctionId} hidden");
            OnRewardHidden?.Invoke(GetImpression(info));
        }

        #endregion

        #region Raise inter events

        protected virtual void RaiseInterClosed(IronSourceAdInfo info)
        {
            _logger.Log($"Interstitial video {info.auctionId} closed. NetworkName: {info.adNetwork}; AdUnitIdentifier: {info.adUnit}; InstanceId: {info.instanceId}");
            OnInterShown?.Invoke(GetImpression(info));
        }

        private void RaiseInterLoaded(IronSourceAdInfo info)
        {
            _logger.Log($"Interstitial video {info.auctionId} loaded. NetworkName: {info.adNetwork}; AdUnitIdentifier: {info.adUnit}; InstanceId: {info.instanceId}");
            _interInfo = info;
            OnInterLoaded?.Invoke();
        }

        private void RaiseInterLoadFailed(IronSourceError error)
        {
            _logger.LogError($"InterVideo - Can't load interstitial video. Error: {error}");
            OnInterLoadFailed?.Invoke(error.ToString());
        }

        private void RaiseInterDisplayFailed(IronSourceError error, IronSourceAdInfo info)
        {
            _logger.LogError($"Failed to show InterVideo with id {info.auctionId}");
            OnInterDisplayFailed?.Invoke(error.ToString());
        }

        private void RaiseInterClicked(IronSourceAdInfo info)
        {
            _logger.Log($"InterVideo with id {info.auctionId} clicked");
            OnInterClicked?.Invoke(GetImpression(info));
        }

        private void RaiseInterHidden(IronSourceAdInfo info)
        {
            _logger.Log($"InterVideo with id {info.auctionId} hidden");
            OnInterHidden?.Invoke(GetImpression(info));
        }

        #endregion
    }
}
#endif