#if UNITY_EDITOR && UNITY_ANDROID
using System.Collections.Generic;
using System.IO;
using System.Linq;
using System.Text;
using System.Xml;
using UnityEditor.Android;
using UnityEditor.Build;

namespace Magify.PlayerBuilder
{
    public partial class MagifyBuildUtilities
    {
        private class ManifestPostprocessor : IPostGenerateGradleAndroidProject
        {
            int IOrderedCallback.callbackOrder => MagifyBuildConfig.PostprocessorsCallbackOrder;

            public void OnPostGenerateGradleAndroidProject(string path)
            {
                Log($"{nameof(ManifestPostprocessor)} enabled: {MagifyBuildConfig.AutoPatchManifestOnBuild}");
                if (!MagifyBuildConfig.AutoPatchManifestOnBuild) return;
                Log("Getting path to the manifest");
                var manifestPath = GetManifestPath(path);
                UpdateManifest(manifestPath);
            }
        }

        static partial void UpdateManifestPartial(string manifestPath)
        {
            Log($"{nameof(UpdateManifest)} Started.");
            if (!File.Exists(manifestPath))
            {
                throw new MagifyBuildException($"Invalid path to AndroidManifest.xml: {manifestPath}");
            }

            Log($"Load android manifest at {manifestPath}");
            var androidManifest = AndroidManifest.LoadFromPath(manifestPath);

            Log("Collecting android packages for cross-promo");
            var packages = MagifyBuildConfig.GetAllAndroidPackages();

            Log($"Adding packages to queries (total is {packages.Count}\n{string.Join("\n", packages)}");
            androidManifest.AddPackagesToQueries(packages);

            Log("Save the new manifest");
            androidManifest.Save();

            Log($"Final AndroidManifest:\n{androidManifest.OuterXml}");
            Log($"{nameof(UpdateManifest)} Finished");
        }

        private static string GetManifestPath(string basePath)
        {
            var pathBuilder = new StringBuilder(basePath);
            pathBuilder.Append(Path.DirectorySeparatorChar).Append("src");
            pathBuilder.Append(Path.DirectorySeparatorChar).Append("main");
            pathBuilder.Append(Path.DirectorySeparatorChar).Append("AndroidManifest.xml");
            return pathBuilder.ToString();
        }

        private class AndroidManifest : XmlDocument
        {
            private readonly string _path;
            const string AndroidXmlNamespace = "http://schemas.android.com/apk/res/android";

            public static AndroidManifest LoadFromPath(string path)
            {
                var manifest = new AndroidManifest(path);
                using (var reader = new XmlTextReader(path))
                {
                    reader.Read();
                    manifest.Load(reader);
                }
                var nsMgr = new XmlNamespaceManager(manifest.NameTable);
                nsMgr.AddNamespace("android", AndroidXmlNamespace);
                return manifest;
            }

            private AndroidManifest(string path)
            {
                _path = path;
            }

            public string Save()
            {
                return SaveAs(_path);
            }

            public string SaveAs(string path)
            {
                using var writer = new XmlTextWriter(path, new UTF8Encoding(false))
                {
                    Formatting = Formatting.Indented
                };
                Save(writer);
                return path;
            }

            public void AddPackagesToQueries(HashSet<string> packageNames)
            {
                var manifestNode = SelectSingleNode("/manifest");
                if (manifestNode == null)
                {
                    return;
                }
                var queriesNode = SelectSingleNode("/manifest/queries");
                if (queriesNode == null)
                {
                    // Create the 'queries' node if it doesn't exist
                    queriesNode = CreateElement("queries");
                    manifestNode.AppendChild(queriesNode);
                }

                // Get the existing package names
                var existingPackages = new HashSet<string>(queriesNode.ChildNodes
                    .OfType<XmlElement>()
                    .Select(node => node.GetAttribute("android:name")));

                // Add new package names that are not already present
                foreach (var packageName in packageNames)
                {
                    if (existingPackages.Contains(packageName)) continue;
                    var packageNode = CreateElement("package");
                    var newAttribute = CreateAndroidAttribute("name", packageName);
                    packageNode.Attributes.Append(newAttribute);
                    queriesNode.AppendChild(packageNode);
                }
            }

            private XmlAttribute CreateAndroidAttribute(string key, string value)
            {
                var attr = CreateAttribute("android", key, AndroidXmlNamespace);
                attr.Value = value;
                return attr;
            }
        }
    }
}
#endif