﻿using System;
using System.Collections.Generic;
using System.Linq;
using System.Reflection;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    internal partial class MagifyManagerTests
    {
        [TestOf(typeof(MagifyManager))]
        public class ShutDown
        {
            [Test]
            public void ShutDown_AndThereIsOnlyDefaultFieldValues()
            {
                // Arrange
                MagifyManager.Initialize(GetConfig());

                // Action
                MagifyManager.ShutDown();

                // Assertion
                if (CheckForNoDefaultFields(out var message))
                {
                    Debug.LogError(message);
                    Assert.Fail();
                }
                Debug.Log(message);
                Assert.Pass();
            }

            private static bool CheckForNoDefaultFields(out string logMessage)
            {
                var fields = typeof(MagifyManager).GetFields(BindingFlags.Static | BindingFlags.Public | BindingFlags.NonPublic);
                var notDefaults = new List<(FieldInfo, object Value, object Expected)>();
                foreach (var field in fields)
                {
                    var value = field.GetValue(null);
                    var expected = GetDefault(field.FieldType);
                    if (value != expected)
                    {
                        notDefaults.Add((field, value, expected));
                    }
                }

                if (notDefaults.Any())
                {
                    logMessage = $"Fields that don`t have default values after {nameof(MagifyManager)} reset:\n";
                    foreach (var (field, value, expected) in notDefaults)
                    {
                        logMessage += $"\t{field.FieldType} {field.Name}; Value: {JsonFacade.SerializeObject(value)}, but Expected: {JsonFacade.SerializeObject(expected)}\n";
                    }
                    return true;
                }

                logMessage = "Checked fields:\n";
                foreach (var field in fields)
                {
                    logMessage += $"\t{field.Name}\n";
                }
                return false;
            }

            private static object GetDefault(Type t)
            {
                return typeof(ShutDown)
                    .GetMethod(nameof(GetDefaultGeneric), BindingFlags.Static | BindingFlags.NonPublic)!
                    .MakeGenericMethod(t)
                    .Invoke(null, null);
            }

            private static T GetDefaultGeneric<T>()
            {
                return default(T);
            }
        }

        internal static MagifyConfig GetConfig()
        {
            return new MagifyConfig
            {
                AppNameGP = "test_app_name",
                AppNameIOS = "test_app_name",
                ConfigPathGP = "JsonModelParse/test_config",
                ConfigPathIOS = "JsonModelParse/test_config",
                UseAdvancedVersion = true,
                EditorDevice = EditorDevice.GetDefaultAndroidDevice(),
                StoragePath = Application.temporaryCachePath,
                CustomClientId = null,
                EditorMoq = null,
                PresenterSettings = null,
                Environment = Environment.Offline,
                SubscriptionStatus = SubscriptionStatus.Inactive,
                AuthorizationStatus = AuthorizationStatus.Authorized,
                IsGdprApplied = false,
                IsLoggingEnabled = false,
                RevenuePerCountryPath = null,
                RevenueLevelsPath = null,
                DefaultCurrencyRatesPath = null,
                SubscriptionMultipliersPath = null,
            };
        }
    }
}