using System;
using System.IO;
using System.Text;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using JetBrains.Annotations;
using NUnit.Framework;
using UnityEngine.Networking;

namespace Magify.Tests
{
    public class StoredAppFeaturesRemoteStorageTests
    {
        private class TestNetwork : IRemoteStorageNetworkClient
        {
            internal delegate (StorageResultCode ResultCode, string Content) Handler(UnityWebRequest request, CancellationToken cancellationToken);

            [CanBeNull]
            private readonly Handler _handler;

            public int? DelayMs { get; set; }

            public TestNetwork([NotNull] Handler handler)
            {
                _handler = handler;
            }

            public async UniTask<StorageResultCode> SendWebRequest(UnityWebRequest request, CancellationToken cancellationToken)
            {
                try
                {
                    if (DelayMs != null)
                    {
                        await UniTask.Delay(TimeSpan.FromMilliseconds(DelayMs.Value), cancellationToken: cancellationToken);
                    }
                    var path = Path.Combine(EditorModeTestsEnvironment.StoredAppFeaturesFolderPath, Path.GetFileName(_url) + ".tmp");
                    var result = _handler!(request, cancellationToken);
                    request.downloadHandler?.Dispose();
                    request.downloadHandler = new DownloadHandlerFile(path + "2");
                    await File.WriteAllTextAsync(path, result.Content, cancellationToken)!;
                    return await (cancellationToken.IsCancellationRequested
                        ? UniTask.FromCanceled<StorageResultCode>(cancellationToken)
                        : UniTask.FromResult(result.ResultCode));
                }
                catch (Exception e)
                {
                    return await UniTask.FromException<StorageResultCode>(e);
                }
            }
        }

        [NotNull]
        private const string Content = "test content";
        [NotNull]
        private static readonly string _url = Path.Combine("test_url.com", "myfilename.txt");
        [NotNull]
        private byte[] ContentBytes => Encoding.UTF8.GetBytes(Content);

        [SetUp, TearDown]
        public void ClearBeforeAndAfterTests()
        {
            if (Directory.Exists(EditorModeTestsEnvironment.StoredAppFeaturesFolderPath))
            {
                Directory.Delete(EditorModeTestsEnvironment.StoredAppFeaturesFolderPath, true);
            }
        }

        [Test]
        [TestCase(StorageResultCode.Success)]
        [TestCase(StorageResultCode.Canceled)]
        [TestCase(StorageResultCode.Timeout)]
        [TestCase(StorageResultCode.StorageError)]
        [TestCase(StorageResultCode.ConnectionError)]
        [TestCase(StorageResultCode.ProtocolError)]
        [TestCase(StorageResultCode.DataProcessingError)]
        [TestCase(StorageResultCode.UnknownError)]
        [TestCase(StorageResultCode.NotPreloaded)]
        [TestCase(StorageResultCode.InvalidAssetError)]
        public async Task EmptyState_ThenLoad_JustGotResultCode(StorageResultCode resultCode)
        {
            // Arrange
            Create(out var storage, new TestNetwork((request, token) => (resultCode, resultCode is StorageResultCode.Success ? Content : null)));

            // Act
            var result = await storage.Load(_url, 30, CancellationToken.None);

            // Assert
            result.Should().NotBeNull();
            result!.Code.Should().Be(resultCode);
        }

        [Test]
        public async Task EmptyState_ThenLoad_GotSuccessWithContent()
        {
            // Arrange
            Create(out var storage, new TestNetwork((request, token) => (StorageResultCode.Success, Content)));

            // Act
            var result = await storage.Load(_url, 30, CancellationToken.None);

            // Assert
            result.Should().NotBeNull();
            result.Code.Should().Be(StorageResultCode.Success);
            result.Value.FilePath.Should().NotBeNull();
            var content = await result.Value.LoadString();
            content.Should().Be(Content);
        }

        [Test]
        [TestCase(StorageResultCode.Canceled)]
        [TestCase(StorageResultCode.Timeout)]
        [TestCase(StorageResultCode.StorageError)]
        [TestCase(StorageResultCode.ConnectionError)]
        [TestCase(StorageResultCode.ProtocolError)]
        [TestCase(StorageResultCode.DataProcessingError)]
        [TestCase(StorageResultCode.UnknownError)]
        [TestCase(StorageResultCode.NotPreloaded)]
        [TestCase(StorageResultCode.InvalidAssetError)]
        public async Task ThereIsOnDisk_LoadingFailedButLoadingFromDiskIsOk(StorageResultCode remoteResultCode)
        {
            // Arrange
            Create(out var storage, new TestNetwork((request, token) => (remoteResultCode, null)));
            storage.SaveToDisk(_url, ContentBytes);

            // Act
            var result = await storage.Load(_url, 30, CancellationToken.None);

            // Assert
            result.Should().NotBeNull();
            result!.Code.Should().Be(StorageResultCode.Success);
        }

        [Test]
        [TestCase(StorageResultCode.Canceled)]
        [TestCase(StorageResultCode.Timeout)]
        [TestCase(StorageResultCode.StorageError)]
        [TestCase(StorageResultCode.ConnectionError)]
        [TestCase(StorageResultCode.ProtocolError)]
        [TestCase(StorageResultCode.DataProcessingError)]
        [TestCase(StorageResultCode.UnknownError)]
        [TestCase(StorageResultCode.NotPreloaded)]
        [TestCase(StorageResultCode.InvalidAssetError)]
        public async Task ThereIsOnDisk_LoadingFailedButLoadedFromDisk(StorageResultCode remoteResultCode)
        {
            // Arrange
            Create(out var storage, new TestNetwork((request, token) => (remoteResultCode, null)));
            storage.SaveToDisk(_url, ContentBytes);

            // Act
            var result = await storage.Load(_url, 30, CancellationToken.None);

            // Assert
            result.Should().NotBeNull();
            result.Code.Should().Be(StorageResultCode.Success);
            result.Value.FilePath.Should().NotBeNull();
            var content = await result.Value.LoadString();
            content.Should().Be(Content);
        }

        [Test]
        public async Task Loaded_MustBeOnDisk_LoadingFailedButLoadedFromDisk()
        {
            // Arrange
            var firstLoaded = false;
            Create(out var storage, new TestNetwork((request, token) =>
            {
                if (firstLoaded)
                    return (StorageResultCode.ConnectionError, null);
                firstLoaded = true;
                return (StorageResultCode.Success, Content);
            }));

            // Act
            using var result1 = await storage.Load(_url, 30, CancellationToken.None);
            using var result2 = await storage.Load(_url, 30, CancellationToken.None);

            // Assert
            await check(result1);
            await check(result2);

            async Task check(ContentHandle<StoredAppFeatureContent> result)
            {
                result.Should().NotBeNull();
                result.Code.Should().Be(StorageResultCode.Success);
                result.Value.FilePath.Should().NotBeNull();
                var content = await result.Value.LoadString();
                content.Should().Be(Content);
            }
        }

        [Test]
        [TestCase(1)]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(200)]
        public async Task RunManyLoaders_TheyHaveToWaitOneRequest(int loaders)
        {
            // Arrange
            var requests = 0;
            var finished = 0;
            var net = new TestNetwork((request, token) =>
            {
                requests++;
                return (StorageResultCode.Success, Content);
            });
            Create(out var storage, net);

            // Act
            net.DelayMs = 1;
            var first = storage.Load(_url, 30, CancellationToken.None);
            net.DelayMs = 100;
            (loaders-1).EnumerateEach().ForEach(_ => storage.Load(_url, 30, CancellationToken.None).ContinueWith(_ => finished++));
            await first;
            finished++;
            await UniTask.Yield();

            // Assert
            requests.Should().Be(1);
            finished.Should().Be(loaders);
        }

        private void Create([NotNull] out StoredAppFeaturesRemoteStorage storage, [NotNull] TestNetwork net)
        {
            storage = new StoredAppFeaturesRemoteStorage(EditorModeTestsEnvironment.RootStoragePath, net);
        }
    }
}