using System;
using System.Collections;
using System.IO;
using System.Linq;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.Networking;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    public class LocalAssetUriBuilder : UriBuilder
    {
        public LocalAssetUriBuilder(string path)
        {
            var localPath = Application.dataPath;
            var index = localPath.LastIndexOf("Assets", StringComparison.InvariantCulture);
            localPath = localPath.Remove(index);

            Scheme = "file";
            Path = localPath + path;
        }
    }

    [Ignore("Need add SetUp with build bundles or build bundles on Unity start")]
    public class StorageTests : MonoBehaviour
    {
        [SetUp]
        [TearDown]
        public void CleanStorageBetweenTests()
        {
            EditorModeTestsEnvironment.Clear();
        }

        [Test]
        public void TestCacheImage()
        {
            // Arrange
            var storage = new TextureRemoteStorage(EditorModeTestsEnvironment.RootStoragePath);
            var testTexture = new Texture2D(2, 2);
            var url = "Assets/Content/Creatives/Images/helloween_badge.png";

            // Act
            storage.SaveToDisk(url, testTexture.GetRawTextureData());

            // Assert
            var isCached = storage.IsCachedOnDisk(url);
            var imagePath = storage.GetCachePath(url);
            if (File.Exists(imagePath))
            {
                File.Delete(imagePath);
            }

            Assert.IsTrue(isCached);
        }

        [UnityTest]
        public IEnumerator TestCacheBundle()
        {
            // Arrange
            var rootGameObject = new GameObject();
            var storage = new BundleRemoteStorage(rootGameObject.transform, EditorModeTestsEnvironment.RootStoragePath);
            var url = new LocalAssetUriBuilder("Bundles/Editor/farmjam_allinone_badge_creative/en.bundle").ToString();
            using var www = UnityWebRequest.Get(url);
            yield return www.SendWebRequest();

            // Act
            storage.SaveToDisk(url, www.downloadHandler.data);

            // Clean up
            var isCached = storage.IsCachedOnDisk(url);
            var bundlePath = storage.GetCachePath(url);
            if (File.Exists(bundlePath))
            {
                File.Delete(bundlePath);
            }

            // Assert
            Assert.IsTrue(isCached);
        }

        [UnityTest]
        public IEnumerator TestLoadCachedBundle()
        {
            // Arrange
            var rootGameObject = new GameObject();
            var storage = new BundleRemoteStorage(rootGameObject.transform, EditorModeTestsEnvironment.RootStoragePath);
            var url = new LocalAssetUriBuilder("Bundles/Editor/farmjam_allinone_badge_creative/en.bundle").ToString();
            using var www = UnityWebRequest.Get(url);
            yield return www.SendWebRequest();
            storage.SaveToDisk(url, www.downloadHandler.data);
            var bundlePath = storage.GetCachePath(url);

            // Act
            var assetBundle = AssetBundle.LoadFromFile(bundlePath);
            var result = assetBundle != null;

            // Clean up
            if (assetBundle != null)
            {
                assetBundle.Unload(true);
            }

            if (File.Exists(bundlePath))
            {
                File.Delete(bundlePath);
            }

            // Assert
            Assert.IsTrue(result);
        }

        [UnityTest]
        public IEnumerator WhenLoadObjectFromBundleTwice_ThenObjectShouldHaveSameReference()
        {
            // Arrange
            var rootGameObject = new GameObject();
            var storage = new BundleRemoteStorage(rootGameObject.transform, EditorModeTestsEnvironment.RootStoragePath);
            var url = new LocalAssetUriBuilder("Bundles/Editor/farmjam_allinone_badge_creative/en.bundle").ToString();
            using var www = UnityWebRequest.Get(url);
            yield return www.SendWebRequest();
            storage.SaveToDisk(url, www.downloadHandler.data);
            var bundlePath = storage.GetCachePath(url);
            var assetBundle = AssetBundle.LoadFromFile(bundlePath);

            // Act
            var firstContent = assetBundle.LoadAllAssets<GameObject>().FirstOrDefault();
            var secondContent = assetBundle.LoadAllAssets<GameObject>().FirstOrDefault();
            var result = firstContent == secondContent;
            if (assetBundle != null)
            {
                assetBundle.Unload(true);
            }

            // Assert
            Assert.IsTrue(result);
        }

        [Test]
        public void WhenLoadBundleFromFile_AndFileAreNotBundle_ThenLogErrorAndBundleIsNull()
        {
            // Ignore log because test failed
            LogAssert.ignoreFailingMessages = true;

            // Arrange
            var storage = new TextureRemoteStorage(EditorModeTestsEnvironment.RootStoragePath);
            var testTexture = new Texture2D(2, 2);
            var url = "Assets/Content/Creatives/Images/helloween_badge.png";
            storage.SaveToDisk(url, testTexture.GetRawTextureData());
            var imagePath = storage.GetCachePath(url);

            // Act
            var assetBundle = AssetBundle.LoadFromFile(imagePath);
            var result = assetBundle == null;

            // Clean up
            if (assetBundle != null)
            {
                assetBundle.Unload(true);
            }

            if (File.Exists(imagePath))
            {
                File.Delete(imagePath);
            }

            // Assert
            Assert.IsTrue(result);
        }

        [UnityTest]
        public IEnumerator WhenLoadTwoDifferentBundleFromFiles_ThenBundlesShouldLoaded()
        {
            // Arrange
            var rootGameObject = new GameObject();
            var storage = new BundleRemoteStorage(rootGameObject.transform, EditorModeTestsEnvironment.RootStoragePath);
            var creativeUrl = new LocalAssetUriBuilder("Bundles/Editor/farmjam_allinone_badge_creative/en.bundle").ToString();
            var prefabUrl = new LocalAssetUriBuilder("Bundles/Editor/farmjam_allinone_badge_prefab/en.bundle").ToString();

            using (var www = UnityWebRequest.Get(creativeUrl))
            {
                yield return www.SendWebRequest();
                storage.SaveToDisk(creativeUrl, www.downloadHandler.data);
            }

            using (var www = UnityWebRequest.Get(prefabUrl))
            {
                yield return www.SendWebRequest();
                storage.SaveToDisk(prefabUrl, www.downloadHandler.data);
            }

            var firstPath = storage.GetCachePath(creativeUrl);
            var secondPath = storage.GetCachePath(prefabUrl);

            // Act
            var firstAssetBundle = AssetBundle.LoadFromFile(firstPath);
            var secondAssetBundle = AssetBundle.LoadFromFile(secondPath);
            var firstResult = firstAssetBundle != null;
            var secondResult = secondAssetBundle != null;

            // Clean up
            if (firstAssetBundle != null)
            {
                firstAssetBundle.Unload(true);
            }

            if (secondAssetBundle != null)
            {
                secondAssetBundle.Unload(true);
            }

            // Assert
            Assert.IsTrue(firstResult);
            Assert.IsTrue(secondResult);
        }
    }
}