using System.Linq;
using System.Collections.Generic;
using System.Diagnostics.CodeAnalysis;
using FluentAssertions;
using Magify.Rx;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class SubjectTests
    {
        [Test]
        public void WhenOnNextToSubject_ThenListenerNotified_AndValueOk()
        {
            // Arrange
            const string targetValue = "hello!";
            var received = default(object);
            using var subject = new Subject<object>();
            using var _ = subject.Subscribe(v => received = v);

            // Act
            subject.OnNext(targetValue);

            // Assert
            received.Should()!.Be(targetValue);
        }

        [Test]
        public void WhenSubjectCreated_ThenListenerAreNotNotified()
        {
            // Arrange
            var received = false;
            using var subject = new Subject<object>();

            // Act
            using var _ = subject.Subscribe(_ => received = true);

            // Assert
            received.Should()!.BeFalse();
        }

        [Test]
        [TestCase(2)]
        [TestCase(10)]
        [TestCase(100)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void WhenOnNextToSubject_ThenManyListenersNotified_AndValueOk(int listenersNumber)
        {
            // Arrange
            const string targetValue = "hello!";
            var received = new object[listenersNumber];
            using var subject = new Subject<object>();
            using var disposables = new CompositeDisposable(listenersNumber);
            listenersNumber.EnumerateEach().ForEach(i => subject.Subscribe(v => received[i] = v).AddTo(disposables));

            // Act
            subject.OnNext(targetValue);

            // Assert
            received.Should()!.AllBeEquivalentTo(targetValue);
        }

        [Test]
        [TestCase(2, 2)]
        [TestCase(10, 10)]
        [TestCase(100, 100)]
        [SuppressMessage("ReSharper", "AccessToDisposedClosure")]
        public void WhenOnNextMultipleTimesToSubject_ThenManyListenersNotified_AndValuesHaveAll_PlusFirstNull(int listenersNumber, int valuesNum)
        {
            // Arrange
            var received = new List<object>[listenersNumber];
            using var subject = new Subject<object>();
            using var disposables = new CompositeDisposable(listenersNumber);
            listenersNumber.EnumerateEach().ForEach(i => subject.Subscribe(v => (received[i] ??= new List<object>()).Add(v)).AddTo(disposables));
            var rnd = new System.Random();
            var values = valuesNum.EnumerateEach().Select(_ => (object)rnd.Next()).ToArray();

            // Act
            values.ForEach(v => subject.OnNext(v));

            // Assert
            received.ForEach(vals => vals.Should()!.ContainInOrder(values));
        }
    }
}