using System;
using System.Linq;
using System.Collections.Generic;
using FluentAssertions;
using NUnit.Framework;
using JetBrains.Annotations;
using Magify.Rx;
using UnityEngine;

namespace Magify.Tests
{
    internal class SelectTests
    {
        [NotNull, ItemNotNull]
        private static object[] _selectors =
        {
            new object[] { (Func<object, object>)(s => ((string)s).Length), "abc", 3 },
            new object[] { (Func<object, object>)(n => Mathf.Pow((int)n, 2)), 2, 4 },
        };

        [NotNull, ItemNotNull]
        private static object[] _manySelectors = _selectors.Select(o => (object[])o).Select(arr => arr.Append(100).ToArray()).ToArray();

        [Test]
        [TestCaseSource(nameof(_selectors))]
        public void WhenWhereApplied_AndNext_ThenCallback([NotNull] Func<object, object> selector, [CanBeNull] object input, [CanBeNull] object expected)
        {
            // Arrange
            var selected = default(object);
            using var subject = new Subject<object>();
            using var _ = subject.Select(selector).Subscribe(s => selected = s);

            // Act
            subject.OnNext(input);

            // Assert
            selected.Should()!.Be(expected);
        }

        [Test]
        [TestCaseSource(nameof(_manySelectors))]
        public void WhenWhereApplied_AndManyNext_ThenManyCallbacks([NotNull] Func<object, object> selector, [CanBeNull] object input, [CanBeNull] object expected, int repeats)
        {
            // Arrange
            var selected = new List<object>(repeats);
            using var subject = new Subject<object>();
            using var _ = subject.Select(selector).Subscribe(s => selected.Add(s));

            // Act
            repeats.EnumerateEach().ForEach(_ => subject.OnNext(input));

            // Assert
            selected.Count.Should()!.Be(repeats);
            selected.Should()!.AllBeEquivalentTo(expected);
        }
    }
}