using System;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    public class LtoAppVersionLimitsTests
    {
        [Test]
        public void IsWithinAppVersionLimits_AndAllLimitsEmpty_ThenShouldReturnTrue()
        {
            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(null)
                .Build();

            var appVersion = "1.0.0";
            if (appVersion == null) throw new ArgumentNullException(nameof(appVersion));
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeTrue();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndLimitValuesEmpty_ThenShouldReturnTrue()
        {
            var limits = new VersionLimit
            {
                VersionFrom = null,
                VersionTo = null,
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "1.0.0";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeTrue();
        }

        [Test]
        public void IsWithinAppVersionLimits_AppVersionInvalid_ThenShouldReturnFalse()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "1.0.0",
                VersionTo = "2.0.0",
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "1-rc";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeFalse();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndStartLimitValueInvalid_ThenShouldReturnFalse()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "aaa"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "1.0.0";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeFalse();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndEndLimitValueInvalid_ThenShouldReturnFalse()
        {
            var limits = new VersionLimit
            {
                VersionTo = "1.1.1.1.1.1.1.1"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "1.0.0";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeFalse();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndAppVersionBelowLimit_ThenShouldReturnFalse()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "2.10.3",
                VersionTo = "2.11.0"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "2.10.0";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeFalse();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndAppVersionAboveLimit_ThenShouldReturnFalse()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "3.1.30",
                VersionTo = "4.0.0.0"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "5.10.2";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeFalse();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndAppVersionEqualLowerLimit_ThenShouldReturnTrue()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "2.10.3"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "2.10.3";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeTrue();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndAppVersionEqualUpperLimit_ThenShouldReturnTrue()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "2.10.3.1",
                VersionTo = "3.3.3.3"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "3.3.3.3";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeTrue();
        }

        [Test]
        public void IsWithinAppVersionLimits_AndAppVersionInsideLimits_ThenShouldReturnTrue()
        {
            var limits = new VersionLimit
            {
                VersionFrom = "2.0.0",
                VersionTo = "2.0.0.2"
            };

            var campaign = new CampaignModelBuilder()
                .WithAppVersionLimit(limits)
                .Build();

            const string appVersion = "2.0.0.1";
            campaign.IsWithinAppVersionLimits(appVersion).Should().BeTrue();
        }
    }
}