using System.Collections.Generic;
using FluentAssertions;
using Newtonsoft.Json;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    public class JsonFacadeTests
    {
        private class TestObject
        {
            public string Name { get; set; }
            public int Value { get; set; }
        }

        [Test]
        public void SerializeObject_ThenShouldCorrectSerializeObjectToString()
        {
            // Arrange
            var testField = new TestObject { Name = null, Value = 1 };

            // Act
            var serializeObject = JsonFacade.SerializeObject(testField);
            Debug.Log(serializeObject);

            // Assert
            serializeObject.Should().ContainAll("Value", "1");
            serializeObject.Should().NotContainAll("Name", "null");
        }

        [Test]
        public void DeserializeObject_ThenShouldCorrectSerializeObjectToString()
        {
            // Arrange
            var testField = new TestObject { Name = null, Value = 1 };
            var serializeObject = JsonFacade.SerializeObject(testField);

            // Act
            var deserializeObject = JsonFacade.DeserializeObject<TestObject>(serializeObject);

            // Assert
            deserializeObject.Name.Should().BeNull();
            deserializeObject.Value.Should().Be(1);
        }

        [Test]
        public void SerializeObject_WithIgnoreNullValueHandling_ThenShouldSerializeWithoutNullFields()
        {
            // Arrange
            var testField = new TestObject { Name = null, Value = 1 };
            JsonFacade.DefaultSettings.NullValueHandling = NullValueHandling.Ignore;

            // Act
            var serializeObject = JsonFacade.SerializeObject(testField);
            Debug.Log(serializeObject);

            // Assert
            serializeObject.Should().NotContainAll("Name", "null");
            serializeObject.Should().ContainAll("Value", "1");
        }

        [Test]
        public void SerializeEmptyDictionary_ThenShouldSerializeAsEmptyJsonString()
        {
            // Arrange
            var testDictionary = new Dictionary<string, object>();

            // Act
            var serializeObject = JsonFacade.SerializeObject(testDictionary);
            Debug.Log(serializeObject);

            // Assert
            serializeObject.Should().Contain("{}");
        }

        [Test]
        public void DeserializeEmptyJson_ThenShouldDeserializeAsEmptyDictionary()
        {
            // Arrange
            const string testJsonString = "{}";

            // Act
            var deserializeObject = JsonFacade.DeserializeObject<Dictionary<string, object>>(testJsonString);

            // Assert
            deserializeObject.Should().BeEmpty();
        }

        [Test]
        public void DeserializeEmptyString_ThenShouldDeserializeAsNullObject()
        {
            // Arrange
            const string testJsonString = "";

            // Act
            var deserializeObject = JsonFacade.DeserializeObject<Dictionary<string, object>>(testJsonString);

            // Assert
            deserializeObject.Should().BeNull();
        }
    }
}