using System;
using FluentAssertions;
using Newtonsoft.Json;
using NUnit.Framework;

namespace Magify.Tests.Json
{
    public static class TimeSpanExtensions
    {
        public static long AsUnis(this TimeSpan span, TimeUnit unit)
        {
            var result = unit switch
            {
                TimeUnit.Ticks => span.Ticks,
                TimeUnit.Millisecond => span.TotalMilliseconds,
                TimeUnit.Second => span.TotalSeconds,
                TimeUnit.Minute => span.TotalMinutes,
                TimeUnit.Hour => span.TotalHours,
                TimeUnit.Day => span.TotalDays,
                _ => throw new ArgumentOutOfRangeException(nameof(unit), unit, null)
            };
            return (long)result;
        }

        public static long AsUnis(this TimeSpan? dt, TimeUnit unit)
        {
            return dt?.AsUnis(unit) ?? 0;
        }
    }

    public class UnitTimeSpanConverterTests
    {
        class NullableDateTimeContainer
        {
            [JsonProperty("time_ticks")]
            [JsonConverter(typeof(UnitTimeSpanConverter))]
            public TimeSpan? TimeInTicks { get; set; }

            [JsonProperty("time_milliseconds")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Millisecond)]
            public TimeSpan? TimeInMillisecond { get; set; }

            [JsonProperty("time_seconds")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Second)]
            public TimeSpan? TimeInSeconds { get; set; }

            [JsonProperty("time_minutes")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Minute)]
            public TimeSpan? TimeInMinutes { get; set; }

            [JsonProperty("time_hours")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Hour)]
            public TimeSpan? TimeInHours { get; set; }

            [JsonProperty("time_days")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Day)]
            public TimeSpan? TimeInDays { get; set; }
        }

        class DateTimeContainer
        {
            [JsonProperty("time_ticks")]
            [JsonConverter(typeof(UnitTimeSpanConverter))]
            public TimeSpan TimeInTicks { get; set; }

            [JsonProperty("time_milliseconds")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Millisecond)]
            public TimeSpan TimeInMillisecond { get; set; }

            [JsonProperty("time_seconds")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Second)]
            public TimeSpan TimeInSeconds { get; set; }

            [JsonProperty("time_minutes")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Minute)]
            public TimeSpan TimeInMinutes { get; set; }

            [JsonProperty("time_hours")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Hour)]
            public TimeSpan TimeInHours { get; set; }

            [JsonProperty("time_days")]
            [JsonConverter(typeof(UnitTimeSpanConverter), TimeUnit.Day)]
            public TimeSpan TimeInDays { get; set; }
        }

        [Test]
        public void WhenValidDateTimeSerialized_AndDeserialized_ThenDateTimeIsCorrect()
        {
            // Arrange
            var now = DateTime.UtcNow - DateTime.UnixEpoch;
            var data = new DateTimeContainer
            {
                TimeInTicks = now,
                TimeInMillisecond = now,
                TimeInSeconds = now,
                TimeInMinutes = now,
                TimeInHours = now,
                TimeInDays = now,
            };
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<DateTimeContainer>(json);

            // Assert
            data.TimeInTicks.Should().Be(now);
            data.TimeInMillisecond.AsUnis(TimeUnit.Millisecond).Should().Be(now.AsUnis(TimeUnit.Millisecond));
            data.TimeInSeconds.AsUnis(TimeUnit.Second).Should().Be(now.AsUnis(TimeUnit.Second));
            data.TimeInMinutes.AsUnis(TimeUnit.Minute).Should().Be(now.AsUnis(TimeUnit.Minute));
            data.TimeInHours.AsUnis(TimeUnit.Hour).Should().Be(now.AsUnis(TimeUnit.Hour));
            data.TimeInDays.AsUnis(TimeUnit.Day).Should().Be(now.AsUnis(TimeUnit.Day));
        }

        [Test]
        public void WhenNullableDateTimeSerialized_AndDeserialized_ThenDateTimeNull()
        {
            // Arrange
            var data = new NullableDateTimeContainer();
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            data.TimeInTicks.Should().BeNull();
            data.TimeInMillisecond.Should().BeNull();
            data.TimeInSeconds.Should().BeNull();
            data.TimeInMinutes.Should().BeNull();
            data.TimeInHours.Should().BeNull();
            data.TimeInDays.Should().BeNull();
        }

        [Test]
        public void WhenDefaultDateTimeSerialized_AndDeserialized_ThenDateIsDefault()
        {
            // Arrange
            var data = new DateTimeContainer();
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<DateTimeContainer>(json);

            // Assert
            data.TimeInTicks.Should().Be(default);
            data.TimeInMillisecond.Should().Be(default);
            data.TimeInSeconds.Should().Be(default);
            data.TimeInMinutes.Should().Be(default);
            data.TimeInHours.Should().Be(default);
            data.TimeInDays.Should().Be(default);
        }

        [Test]
        public void WhenValidNullableDateTimeSerialized_AndDeserialized_ThenDateTimeIsCorrect()
        {
            // Arrange
            var now = DateTime.UtcNow - DateTime.UnixEpoch;
            var data = new NullableDateTimeContainer
            {
                TimeInTicks = now,
                TimeInMillisecond = now,
                TimeInSeconds = now,
                TimeInMinutes = now,
                TimeInHours = now,
                TimeInDays = now,
            };
            var json = JsonFacade.SerializeObject(data);

            // Act
            data = JsonFacade.DeserializeObject<NullableDateTimeContainer>(json);

            // Assert
            data.TimeInTicks.Should().NotBeNull();
            data.TimeInMillisecond.Should().NotBeNull();
            data.TimeInSeconds.Should().NotBeNull();
            data.TimeInMinutes.Should().NotBeNull();
            data.TimeInHours.Should().NotBeNull();
            data.TimeInDays.Should().NotBeNull();

            data.TimeInTicks.Should().Be(now);
            data.TimeInMillisecond.AsUnis(TimeUnit.Millisecond).Should().Be(now.AsUnis(TimeUnit.Millisecond));
            data.TimeInSeconds.AsUnis(TimeUnit.Second).Should().Be(now.AsUnis(TimeUnit.Second));
            data.TimeInMinutes.AsUnis(TimeUnit.Minute).Should().Be(now.AsUnis(TimeUnit.Minute));
            data.TimeInHours.AsUnis(TimeUnit.Hour).Should().Be(now.AsUnis(TimeUnit.Hour));
            data.TimeInDays.AsUnis(TimeUnit.Day).Should().Be(now.AsUnis(TimeUnit.Day));
        }
    }
}