using System;
using FluentAssertions;
using Newtonsoft.Json;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests.Json
{
    public class DayOfWeekConverterTests
    {
        class NullableDayOfWeekContainer
        {
            [JsonProperty("day_of_week")]
            [JsonConverter(typeof(DayOfWeekConverter))]
            public DayOfWeek? DayOfWeek { get; set; }
        }

        class DayOfWeekContainer
        {
            [JsonProperty("day_of_week")]
            [JsonConverter(typeof(DayOfWeekConverter))]
            public DayOfWeek DayOfWeek { get; set; }
        }

        [Test]
        public void WhenValidDayOfWeekSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var period = new DayOfWeekContainer
            {
                DayOfWeek = DayOfWeek.Monday,
            };
            var json = JsonFacade.SerializeObject(period);

            // Act
            period = JsonFacade.DeserializeObject<DayOfWeekContainer>(json);

            // Assert
            period.DayOfWeek.Should().Be(DayOfWeek.Monday);
        }

        [Test]
        public void WhenValidNullableDayOfWeekSerialized_AndDeserialized_ThenDayOfWeekIsCorrect()
        {
            // Arrange
            var period = new NullableDayOfWeekContainer
            {
                DayOfWeek = DayOfWeek.Monday,
            };
            var json = JsonFacade.SerializeObject(period);

            // Act
            period = JsonFacade.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            period.DayOfWeek.Should().NotBeNull();
            period.DayOfWeek.Should().Be(DayOfWeek.Monday);
        }

        [Test]
        public void WhenValidJsonDeserialized_WithLowerCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"mon\"}";

            // Act
            var container = JsonConvert.DeserializeObject<DayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Monday);
        }

        [Test]
        public void WhenValidNullableJsonDeserialized_WithLowerCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"mon\"}";

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Monday);
        }

        [Test]
        public void WhenValidJsonDeserialized_WithUpperCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"SUN\"}";

            // Act
            var container = JsonConvert.DeserializeObject<DayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Sunday);
        }

        [Test]
        public void WhenValidNullableJsonDeserialized_WithUpperCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"SUN\"}";

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Sunday);
        }

        [Test]
        public void WhenValidJsonDeserialized_WithMixedCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"tUe\"}";

            // Act
            var container = JsonConvert.DeserializeObject<DayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Tuesday);
        }

        [Test]
        public void WhenValidNullableJson_WithMixedCaseDayOfWeek_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"tUe\"}";

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().Be(DayOfWeek.Tuesday);
        }

        [Test]
        public void WhenInvalidJsonDeserialized_AndDayOfWeekNotNullable_ThenDeserializationFailed()
        {
            // Arrange
            var json = "{\"day_of_week\":\"INVALID\"}";

            // Act
            Assert.Throws<JsonSerializationException>(() => { JsonConvert.DeserializeObject<DayOfWeekContainer>(json); });
        }

        [Test]
        public void WhenInvalidJsonDeserialized_AndDayOfWeekNullable_ThenDeserializationCorrect()
        {
            // Arrange
            var json = "{\"day_of_week\":\"INVALID\"}";
            LogAssert.Expect(LogType.Exception, "JsonSerializationException: Invalid day of week value.");

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().BeNull();
        }

        [Test]
        public void WhenValidNullableJson_WithNullDayOfWeek_ThenDeserializationCorrectWithNullResult()
        {
            // Arrange
            var json = "{\"day_of_week\":null}";

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().BeNull();
        }

        [Test]
        public void WhenValidNullableJson_WithMissingDayOfWeekField_ThenDeserializationCorrectWithNullResult()
        {
            // Arrange
            var json = "{}";

            // Act
            var container = JsonConvert.DeserializeObject<NullableDayOfWeekContainer>(json);

            // Assert
            container.DayOfWeek.Should().BeNull();
        }
    }
}