﻿using System;
using System.Linq;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    internal class DownloadPromiseTests
    {
        [Test]
        [TestCase(StorageResultCode.Success)]
        [TestCase(StorageResultCode.Timeout)]
        [TestCase(StorageResultCode.DataProcessingError)]
        public void CreatedHandle_AndSetResult_ExpectedCorrectValue(StorageResultCode expectedResult)
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var result = default(StorageResultCode);
            var handle = promise.GetHandle();
            handle.Task.ContinueWith(resultCode => result = resultCode).Forget();

            // Act
            promise.TrySetResult(expectedResult);

            // Assert
            result.Should().Be(expectedResult);
        }

        [Test]
        [TestCase(StorageResultCode.Success, 1)]
        [TestCase(StorageResultCode.Timeout, 1)]
        [TestCase(StorageResultCode.Success, 2)]
        [TestCase(StorageResultCode.Timeout, 2)]
        [TestCase(StorageResultCode.Success, 10)]
        [TestCase(StorageResultCode.Timeout, 10)]
        [TestCase(StorageResultCode.Success, 100)]
        [TestCase(StorageResultCode.Timeout, 100)]
        public void CreatedManyHandles_AndSetResult_ExpectedCorrectValue(StorageResultCode expectedResult, int awaiters)
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var result = new StorageResultCode?[awaiters];
            for (var i = 0; i < awaiters; i++)
            {
                var index = i;
                var handle = promise.GetHandle();
                handle.Task.ContinueWith(resultCode => result[index] = resultCode).Forget();
            }

            // Act
            promise.TrySetResult(expectedResult);

            // Assert
            result.ForEach(r => r.Should().HaveValue());
            result.ForEach(r => r!.Value.Should().Be(expectedResult));
        }

        [Test]
        [TestCase(StorageResultCode.Success, 1)]
        [TestCase(StorageResultCode.Timeout, 1)]
        [TestCase(StorageResultCode.Success, 2)]
        [TestCase(StorageResultCode.Timeout, 2)]
        [TestCase(StorageResultCode.Success, 10)]
        [TestCase(StorageResultCode.Timeout, 10)]
        [TestCase(StorageResultCode.Success, 100)]
        [TestCase(StorageResultCode.Timeout, 100)]
        public void CreatedManyHandles_AndDisposeHandles_ExpectedNoValue(StorageResultCode expectedResult, int awaiters)
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var result = new StorageResultCode?[awaiters];
            var handles = new DownloadPromise<StorageResultCode>.Handle[awaiters];
            for (var i = 0; i < awaiters; i++)
            {
                var index = i;
                var handle = handles[index] = promise.GetHandle();
                handle.Task.ContinueWith(resultCode => result[index] = resultCode).Forget();
            }

            // Act
            handles.ForEach(h => h.Dispose());

            // Assert
            result.ForEach(r => r.Should().NotHaveValue());
        }

        [Test]
        [TestCase(StorageResultCode.Success, 2, 1)]
        [TestCase(StorageResultCode.Timeout, 2, 1)]
        [TestCase(StorageResultCode.Success, 10, 7)]
        [TestCase(StorageResultCode.Timeout, 10, 7)]
        [TestCase(StorageResultCode.Success, 100, 69)]
        [TestCase(StorageResultCode.Timeout, 100, 69)]
        public void CreatedManyHandles_AndDisposeSomeHandles_ExpectedNoValue(StorageResultCode expectedResult, int awaiters, int disposeAwaiters)
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var result = new StorageResultCode?[awaiters];
            var handles = new DownloadPromise<StorageResultCode>.Handle[awaiters];
            for (var i = 0; i < awaiters; i++)
            {
                var index = i;
                var handle = handles[index] = promise.GetHandle();
                handle.Task.ContinueWith(resultCode => result[index] = resultCode).Forget();
            }

            // Act
            var shuffled = handles.ToList();
            shuffled.Sort((_, _) => Guid.NewGuid().CompareTo(Guid.NewGuid()));
            shuffled.Take(disposeAwaiters).ForEach(h => h.Dispose());
            promise.TrySetResult(expectedResult);

            // Assert
            result.Count(r => !r.HasValue).Should().Be(disposeAwaiters);
            result.Count(r => r.HasValue).Should().Be(awaiters - disposeAwaiters);
            result.Where(r => r.HasValue).ForEach(r => r!.Value.Should().Be(expectedResult));
        }

        private static object[] HandlesNumberCases()
        {
            var random = new Random();
            return 10.EnumerateEach().Select(_ => (object)(random.Next() % 200)).Where(n => n is not 0).Append(1).ToArray();
        }

        [Test]
        public void CreateHandleAndDispose_ThenPromiseIsCancelled()
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var handle = promise.GetHandle();

            // Act
            handle.Dispose();

            // Assert
            promise.IsCancellationRequested.Should().BeTrue();
        }

        [Test]
        [TestCaseSource(nameof(HandlesNumberCases))]
        public void CreateManyHandlesAndDispose_ThenPromiseIsCancelled(int handlesNumber)
        {
            // Arrange
            var promise = new DownloadPromise<StorageResultCode>();
            var handles = handlesNumber.EnumerateEach().Select(_ => promise.GetHandle()).ToArray();

            // Act
            handles.ForEach(h => h!.Dispose());

            // Assert
            promise.IsCancellationRequested.Should().BeTrue();
        }
    }
}