﻿using System;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;
using UnityEngine.TestTools;

namespace Magify.Tests
{
    internal static partial class ServerApiTests
    {
        public class Token : ServerApiTestsPart
        {
            [Test]
            public async Task WhenGetAuthToken_AndCancelAllTokenLoadings_ThenGetOperationCancelledException()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        await serverApi.CancelAllTokenLoadings();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.GetAuthorizationTokenAsync(cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            [TestCase(2)]
            [TestCase(5)]
            [TestCase(100)]
            public async Task WhenMultiplyLoading_ThenHaveCorrectResult(int loadingsNumber)
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var shouldCountRequest = false;
                var requestCount = 0;
                var promise = new UniTaskCompletionSource();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    // ReSharper disable once AccessToModifiedClosure
                    if (shouldCountRequest) requestCount++;
                    promise.TrySetResult();
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    shouldCountRequest = true;
                    loadingsNumber.EnumerateEach().ForEach(_ => serverApi.GetAuthorizationTokenAsync(cancellationToken).AsUniTask().Forget());
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                    return;
                }

                await promise.Task;
                await UniTask.Yield();

                // Assert
                requestCount.Should()!.Be(1);
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenServerApiDisposed_ThenAuthTokenIsCancelled()
            {
                // Arrange
                var exception = default(OperationCanceledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                var systems = Create(network, out var serverApi);
                var disposable = systems as IDisposable;
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    UniTask.RunOnThreadPool(async () =>
                    {
                        await UniTask.Delay(2, cancellationToken: cancellationToken);
                        await UniTask.SwitchToMainThread();
                        disposable.Dispose();
                    }, cancellationToken: cancellationToken).Forget();
                    await serverApi.GetAuthorizationTokenAsync(cancellationToken);
                }
                catch (OperationCanceledException e)
                {
                    exception = e;
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenGetAuthToken_AndErrorCodeIsExpired_ThenGetError()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var firstRequestPassed = false;
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    if (firstRequestPassed) await UniTask.Delay(1000, cancellationToken: token);
                    firstRequestPassed = true;
                    return NetworkMoq.ProtocolErrorResult(message, new ErrorResponse() { Error = new ErrorResponse.ErrorInfo() { Code = ErrorCode.Expired } });
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Pre Assert
                expect();

                // Act
                try
                {
                    _ = serverApi.GetAuthorizationTokenAsync(cancellationToken);
                    await UniTask.WaitUntil(() => firstRequestPassed, cancellationToken: cancellationToken);
                    await UniTask.Yield();
                }
                catch (Exception)
                {
                    // Ignore
                }

                // Assert
                void expect() => LogAssert.Expect(LogType.Error, new Regex(string.Empty));
            }

            [Test]
            public async Task WhenGetAuthToken_AndGetRevokedError_ThenRequestIsNotFinished()
            {
                // Arrange
                var isFinished = false;
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, _, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.ProtocolErrorResult(default, new ErrorResponse() { Error = new ErrorResponse.ErrorInfo() { Code = ErrorCode.Revoked } });
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();
                await serverApi.CancelAllServerInteractions();

                // Act
                try
                {
                    serverApi.GetAuthorizationTokenAsync(cancellationToken).ContinueWith(_ => isFinished = true).AsUniTask().Forget();
                }
                catch (Exception e)
                {
                    Debug.Log(e.Message);
                   // Ignore
                }
                await UniTask.Delay(100, cancellationToken: cancellationToken);

                // Assert
                isFinished.Should()!.BeFalse();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenGetAuthTokenWithCancelledToken_ThenGetException()
            {
                // Arrange
                var exception = default(MagifyAuthTokenLoadingCancelledException);
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();

                // Act
                var cts = CancellationTokenSource.CreateLinkedTokenSource(cancellationToken);
                cts.Cancel();
                try
                {
                    await serverApi.GetAuthorizationTokenAsync(cts.Token);
                }
                catch (MagifyAuthTokenLoadingCancelledException e)
                {
                    exception = e;
                    Debug.Log($"Exception caught: {exception.Message}");
                }

                // Assert
                exception.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenGetAuthToken_AndEndpointUrlIsOffline_ThenTokenShouldBeNull()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi, EndpointUrl.Offline);
                systems.InitializeAll();

                // Act
                var authToken = await serverApi.GetAuthorizationTokenAsync(cancellationToken);

                // Assert
                authToken.Should()!.BeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }

            [Test]
            public async Task WhenGetAuthToken_ThenAuthTokenShouldNotBeNull()
            {
                // Arrange
                var cancellationToken = GetCancellationToken();
                var network = new NetworkMoq(async (_, message, token) =>
                {
                    await UniTask.Delay(50, cancellationToken: token);
                    return NetworkMoq.OkResult(message);
                });
                using var systems = Create(network, out var serverApi);
                systems.InitializeAll();

                // Act
                var authToken = await serverApi.GetAuthorizationTokenAsync(cancellationToken);

                // Assert
                authToken.Should()!.NotBeNull();
                cancellationToken.IsCancellationRequested.Should()!.BeFalse();
            }
        }
    }
}