using System.Collections.Generic;
using System.Linq;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    [TestOf(typeof(CampaignUpdateHandler))]
    public class CampaignUpdateHandlerTests
    {
        private const string TestConfig = "JsonModelParse/test_config";
        private const string CampaignName = "UC Sample GP bonus noui";
        private const string EventName = "sample_bonus_noui";
        private const string ProductId = "old_test_bonus_noui";
        private const string ProductName = "old Nested bonus noui";
        private const string EventSource = "tests";

        [Test]
        [UnitTestTarget(typeof(CampaignUpdateHandler), nameof(CampaignUpdateHandler.SubscribeCampaignUpdates))]
        [UnitTestTarget(typeof(CampaignUpdateHandler), nameof(CampaignUpdateHandler.OnProductUsed))]
        public void WhenSubscribeCampaignUpdates_AndCallOnProductUsed_ThenCallbackReceived()
        {
            // Arrange
            using var subsystems = CreateHandler(out var updateHandler, out var campaignsProvider, out var campaignsTracker);
            var callbackReceived = false;
            CreateAndTrackClick<BonusCampaign>(CampaignName, ProductId, ProductName, ProductType.Bonus, campaignsTracker);

            // Act
            updateHandler.SubscribeCampaignUpdates(CampaignName, onUpdate);

            // Load new campaign version;
            var context = CampaignsContext.Deserialize(Resources.Load<TextAsset>(TestConfig).text);
            subsystems.ForEach<IContextListener>(t => t.UpdateContext(context, ContextKind.Default));
            updateHandler.OnProductUsed(ProductId);

            // Assert
            callbackReceived.Should().BeTrue();

            updateHandler.UnsubscribeCampaignUpdates(CampaignName, onUpdate);

            void onUpdate()
            {
                callbackReceived = true;
            }
        }

        [Test]
        [UnitTestTarget(typeof(CampaignUpdateHandler), nameof(CampaignUpdateHandler.SubscribeCampaignUpdates))]
        [UnitTestTarget(typeof(CampaignUpdateHandler), nameof(CampaignUpdateHandler.OnProductUsed))]
        [UnitTestTarget(typeof(CampaignUpdateHandler), nameof(CampaignUpdateHandler.UnsubscribeCampaignUpdates))]
        public void WhenSubscribeCampaignUpdates_AndUnsubscribe_AndCallOnProductUsed_ThenCallbackNotReceived()
        {
            // Arrange
            using var subsystems = CreateHandler(out var updateHandler, out var campaignsProvider, out var campaignsTracker);
            var callbackReceived = false;
            CreateAndTrackClick<BonusCampaign>(CampaignName, ProductId, ProductName, ProductType.Bonus, campaignsTracker);

            // Act
            updateHandler.SubscribeCampaignUpdates(CampaignName, onUpdate);

            // Load new campaign version;
            var context = CampaignsContext.Deserialize(Resources.Load<TextAsset>(TestConfig).text);
            subsystems.ForEach<IContextListener>(t => t.UpdateContext(context, ContextKind.Default));

            updateHandler.UnsubscribeCampaignUpdates(CampaignName, onUpdate);
            updateHandler.OnProductUsed(ProductId);

            // Assert
            callbackReceived.Should().BeFalse();

            void onUpdate()
            {
                callbackReceived = true;
            }
        }

        private static SubsystemsCollection CreateHandler(out CampaignUpdateHandler campaignUpdateHandler, out CampaignsProvider campaignsProvider, out CampaignsTracker campaignsTracker)
        {
            var systems = new SubsystemsCollection();
            var platform = new PlatformDefault().AddTo(systems);
            var appVersionProvider = new AppVersionProvider().AddTo(systems);
            var generalPrefs = GeneralPrefs.Create(EditorModeTestsEnvironment.GeneralPrefsPath, EditorModeTestsEnvironment.LocalGeneralPrefsPath, appVersionProvider).AddTo(systems);
            var offersPrefs = OffersPrefs.Create(EditorModeTestsEnvironment.OffersPrefsName).AddTo(systems);
            var campaignsCollection = new CampaignsCollection().AddTo(systems);
            var countersStorage = new CountersStorage(EditorModeTestsEnvironment.CountersFolderPath).AddTo(systems);
            var counters = new Counters(countersStorage).AddTo(systems);
            var sessionCounter = new SessionCounter(generalPrefs, platform).AddTo(systems);
            var limitsHolder = new LimitsHolder(generalPrefs, sessionCounter).AddTo(systems);
            campaignsTracker = new CampaignsTracker(generalPrefs, counters, limitsHolder, platform).AddTo(systems);
            var ltoCampaignManager = new LtoCampaignManager(campaignsTracker, campaignsCollection, generalPrefs, offersPrefs, platform, counters, EditorModeTestsEnvironment.RootStoragePath, appVersionProvider).AddTo(systems);
            campaignsProvider = new CampaignsProvider(generalPrefs, campaignsTracker, campaignsCollection, counters, limitsHolder, platform, ltoCampaignManager).AddTo(systems);

            campaignUpdateHandler = new CampaignUpdateHandler(campaignsTracker, campaignsProvider, campaignsCollection, platform, generalPrefs, ltoCampaignManager).AddTo(systems);
            return systems;
        }

        private static void CreateAndTrackClick<T>(string campaignName, string productId, string productName, ProductType productType, CampaignsTracker campaignsTracker) where T : ICampaign, new()
        {
            var campaign = new T();
            typeof(T).GetProperty(nameof(ICampaign.Name))?.SetValue(campaign, campaignName);
            var nesteds = new[]
            {
                new NestedCampaign
                {
                    Name = productName,
                    ProductId = productId,
                    Type = productType,
                }
            };
            campaignsTracker.RegisterCampaignRequest(BuildRequest(campaign, nesteds: nesteds));
            campaignsTracker.TrackClick(campaignName, productId);
        }

        private static CampaignRequest BuildRequest(
            ICampaign campaign,
            SourceType source = SourceType.Event,
            IReadOnlyList<NestedCampaign> nesteds = default,
            Dictionary<string, object> @params = default,
            bool isDefault = true)
        {
            return new CampaignRequest
            {
                Campaign = campaign,
                Source = new CampaignSource(EventSource, source),
                IsDefaultCampaign = isDefault,
                NestedCampaigns = nesteds,
                ParsedProducts = new AbstractProductsCollection(nesteds
                    !.Select(n => CampaignParser.ProductsParsers[n.Type!.Value].ParseProduct(n, string.Empty))
                    !.ToList()),
                CustomParams = @params
            };
        }
    }
}