using System.Collections.Generic;
using System.Threading;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;
// ReSharper disable AccessToDisposedClosure

namespace Magify.Tests
{
    internal class RequestContextTests
    {
        [Test]
        [TestCase(1)]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(10)]
        public void WhenGetAllRequestsToken_AndRequestContextDispose_ThenAllRequestsTokenShouldBeCancelled(int loadings)
        {
            //Arrange
            var tokens = new List<CancellationToken>();
            var cancelledTokensCount = 0;
            var requestContext = new RequestContext<object>(default, default, DefaultResponseProcessor);

            //Act
            loadings.EnumerateEach().ForEach(_ =>
            {
                tokens.Add(requestContext.GetAllRequestsToken());
            });
            requestContext.Dispose();

            tokens.ForEach(token =>
            {
                if (token.IsCancellationRequested) cancelledTokensCount++;
            });

            //Assert
            cancelledTokensCount.Should()!.Be(loadings);
        }

        [Test]
        [TestCase(1)]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(10)]
        public void WhenGetActiveRequestCancellationToken_AndRequestContextDispose_ThenActiveRequestTokenShouldBeCancelled(int loadings)
        {
            //Arrange
            var tokens = new List<CancellationToken>();
            var cancelledTokensCount = 0;
            var requestContext = new RequestContext<object>(default, default, DefaultResponseProcessor);

            //Act
            loadings.EnumerateEach().ForEach(_ =>
            {
                tokens.Add(requestContext.GetActiveRequestCancellationToken());
            });
            requestContext.Dispose();

            tokens.ForEach(token =>
            {
                if (token.IsCancellationRequested) cancelledTokensCount++;
            });

            //Assert
            cancelledTokensCount.Should()!.Be(loadings);
        }

        [Test]
        [TestCase(1)]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(10)]
        public void WhenGetAllRequestsToken_AndCancelAllRequests_ThenAllRequestTokenShouldBeCancelled(int loadings)
        {
            //Arrange
            var tokens = new List<CancellationToken>();
            var cancelledTokensCount = 0;
            var requestContext = new RequestContext<object>(default, default, DefaultResponseProcessor);

            //Act
            loadings.EnumerateEach().ForEach(_ =>
            {
                tokens.Add(requestContext.GetAllRequestsToken());
            });
            requestContext.CancelAllRequests();

            tokens.ForEach(token =>
            {
                if (token.IsCancellationRequested) cancelledTokensCount++;
            });

            //Assert
            cancelledTokensCount.Should()!.Be(loadings);
        }

        [Test]
        [TestCase(1)]
        [TestCase(2)]
        [TestCase(5)]
        [TestCase(10)]
        public void WhenGetActiveRequestCancellationToken_AndCancelActiveRequest_ThenActiveRequestTokenShouldBeCancelled(int loadings)
        {
            //Arrange
            var tokens = new List<CancellationToken>();
            var cancelledTokensCount = 0;
            var requestContext = new RequestContext<object>(default, default, DefaultResponseProcessor);

            //Act
            loadings.EnumerateEach().ForEach(_ =>
            {
                tokens.Add(requestContext.GetActiveRequestCancellationToken());
            });
            requestContext.CancelActiveRequest();

            tokens.ForEach(token =>
            {
                if (token.IsCancellationRequested) cancelledTokensCount++;
            });

            //Assert
            cancelledTokensCount.Should()!.Be(loadings);
        }

        [Test]
        [TestCase(true)]
        [TestCase(false)]
        public void WhenTryCreateNewCompletionSource_ThenCheckSingleRequestInProgressShouldSame(bool needCreateNewCompletionSource)
        {
            //Arrange
            var singleModRequestTools = new SingleModeRequestTools<object>();
            var requestContext = new RequestContext<object>(singleModRequestTools, default, DefaultResponseProcessor);

            //Act
            if (needCreateNewCompletionSource)
            {
                singleModRequestTools.TryCreateNewCompletionSource(CancellationToken.None, out UniTask<object>? completionTask, out _);
            }

            //Assert
            requestContext.CheckSingleRequestInProgress().Should()!.Be(needCreateNewCompletionSource);
        }

        private RepeatState DefaultResponseProcessor(WebResponseMessage message, out object o)
        {
            o = default;
            return RepeatState.Finish;
        }
    }
}