﻿using System;
using System.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MinimalMagifyClientTests
    {
        public class Advertisement : MinimalMagifyClientTestsPart
        {
            private static readonly object[] _adsImpressions = new object[]
            {
                (EventType.ApplovinAdsImpression, (IAdsImpression)new ApplovinAdsImpression
                {
                    Id = "applovin_ad_id",
                    Network = "applovin_ad_network",
                    ImpressionId = "applovin_ad_impression_id",
                    Revenue = 123.456d,
                    Precision = "applovin_ad_precision",
                }),
                (EventType.IronSourceAdsImpression, (IAdsImpression)new IronSourceAdsImpression
                {
                    Id = "ironsource_ad_id",
                    Network = "ironsource_ad_network",
                    AdUnit = "ironsource_ad_unit",
                    InstanceName = "ironsource_ad_instance_name",
                    Revenue = 321.654d,
                    Precision = "ironsource_ad_precision",
                    Ab = "ironsource_ad_ab",
                    AuctionId = "ironsource_ad_auction_id",
                    SegmentName = "ironsource_ad_segment_name",
                    LifetimeRevenue = 987.654d,
                    EncryptedCpm = "ironsource_ad_encrypted_cpm",
                }),
            };

            [Test]
            [TestCaseSource(nameof(_adsImpressions))]
            public async Task TrackAdsImpression_HaveAdsEventSent((EventType EventType, IAdsImpression Impression) arg)
            {
                // Arrange
                var got = false;
                var @event = default(AdsImpressionEvent);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (!got) got = NetworkMoq.TryExtractEvent(message, arg.EventType, out @event);
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefaultAndInit(Environment.Staging, network);
                GetAnalyticsTracker(magifyClient).SetupAnalyticsConfig(new AnalyticsConfiguration());

                // Act
                magifyClient.TrackAdsImpression(null, arg.Impression);
                await Task.Delay(100);

                // Assert
                got.Should().Be(true);
                @event.Should().NotBeNull();
                @event.Should().BeOfType<AdsImpressionEvent>();
                switch (arg.EventType)
                {
                    case EventType.ApplovinAdsImpression:
                        @event.AdUnitId.Should().Be(((ApplovinAdsImpression)arg.Impression).Id);
                        @event.NetworkName.Should().Be(((ApplovinAdsImpression)arg.Impression).Network);
                        @event.Precision.Should().Be(((ApplovinAdsImpression)arg.Impression).Precision);
                        @event.PublisherRevenue.Should().Be(((ApplovinAdsImpression)arg.Impression).Revenue);
                        @event.ImpressionId.Should().Be(((ApplovinAdsImpression)arg.Impression).ImpressionId);
                        break;
                    case EventType.IronSourceAdsImpression:
                        @event.InstanceId.Should().Be(((IronSourceAdsImpression)arg.Impression).Id);
                        @event.AdNetwork.Should().Be(((IronSourceAdsImpression)arg.Impression).Network);
                        @event.InstanceName.Should().Be(((IronSourceAdsImpression)arg.Impression).InstanceName);
                        @event.AdUnit.Should().Be(((IronSourceAdsImpression)arg.Impression).AdUnit);
                        @event.Revenue.Should().Be(((IronSourceAdsImpression)arg.Impression).Revenue);
                        @event.Precision.Should().Be(((IronSourceAdsImpression)arg.Impression).Precision);
                        @event.Ab.Should().Be(((IronSourceAdsImpression)arg.Impression).Ab);
                        @event.AuctionId.Should().Be(((IronSourceAdsImpression)arg.Impression).AuctionId);
                        @event.SegmentName.Should().Be(((IronSourceAdsImpression)arg.Impression).SegmentName);
                        @event.LifetimeRevenue.Should().Be(((IronSourceAdsImpression)arg.Impression).LifetimeRevenue);
                        @event.EncryptedCpm.Should().Be(((IronSourceAdsImpression)arg.Impression).EncryptedCpm);
                        break;
                    default:
                        throw new ArgumentOutOfRangeException();
                }
            }
        }
    }
}