﻿using System;
using System.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class MinimalMagifyClientTests
    {
        public class Adjust : MinimalMagifyClientTestsPart
        {
            private const string AdId = "adjust_id";

            [Test]
            public void WhenMagifyClientInit_AdjustIdMustBeNull()
            {
                // Arrange
                // Act
                using var magifyClient = CreateDefaultAndInit();

                // Assert
                magifyClient.AdjustId.Value.Should().BeNull();
            }

            [Test]
            public async Task WhenMagifyClientInit_TokenMustNotHaveAdjustId()
            {
                // Arrange
                var got = false;
                var network = new NetworkMoq((url, message, token) =>
                {
                    got = got || NetworkMoq.TryExtractToken(message, out var authToken) && authToken.AdjustId == null;
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefault(networkClient: network);

                // Act
                magifyClient.InitializeSdk();
                await magifyClient.ForceReissueAuthToken(GetCancellationToken());

                // Assert
                got.Should().BeTrue();
            }

            [Test]
            public async Task WhenMagifyClientInit_TokenMustHaveAdjustId()
            {
                // Arrange
                var got = false;
                var adjustId = default(string);
                var network = new NetworkMoq((url, message, token) =>
                {
                    got |= NetworkMoq.TryExtractToken(message, out var authToken);
                    adjustId ??= authToken?.AdjustId;
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefault(networkClient: network);

                // Act
                magifyClient.InitializeSdk();
                magifyClient.AdjustId.Value = AdId;
                await magifyClient.ForceReissueAuthToken(GetCancellationToken());

                // Assert
                got.Should().BeTrue();
                adjustId.Should().Be("adjust_id");
            }

            [Test]
            public async Task WhenMagifyClientInit_TokenWithoutAdId_Before_WithAdId()
            {
                // Arrange
                var withoutAdIdTime = default(DateTime?);
                var withAdIdTime = default(DateTime?);
                var network = new NetworkMoq((url, message, token) =>
                {
                    if (NetworkMoq.TryExtractToken(message, out var authToken))
                    {
                        if (authToken.AdjustId == null) withoutAdIdTime ??= DateTime.Now;
                        if (authToken.AdjustId != null) withAdIdTime ??= DateTime.Now;
                    }
                    return NetworkMoq.OkResult(message);
                });
                using var magifyClient = CreateDefault(networkClient: network);

                // Act
                magifyClient.InitializeSdk();
                await magifyClient.ForceReissueAuthToken(GetCancellationToken());
                magifyClient.AdjustId.Value = AdId;
                await magifyClient.ForceReissueAuthToken(GetCancellationToken());

                // Assert
                withoutAdIdTime.Should().NotBeNull();
                withAdIdTime.Should().NotBeNull();
                withoutAdIdTime.Should().BeBefore(withAdIdTime.Value);
            }
        }
    }
}