using System;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using NUnit.Framework;
using UnityEngine;

namespace Magify.Tests
{
    internal partial class MagifyClientTests
    {
        public class Sync : MagifyClientTestsPart
        {
            [Test]
            [TestCase(ClientKind.Minimal)]
            [TestCase(ClientKind.Full)]
            public void WhenSdkSetup_ThenOnConfigSyncRequestedCalled(ClientKind kind)
            {
                // Arrange
                using var client = CreateDefault(kind);
                var configRequests = 0;
                client.OnConfigSyncRequested += () => configRequests++;

                // Act
                client.InitializeSdk();
                client.Setup();

                // Assert
                configRequests.Should().Be(1);
            }

            [Test]
            [TestCase(ClientKind.Minimal)]
            [TestCase(ClientKind.Full)]
            public void WhenSetMediaSource_ThenOnConfigSyncRequestedCalled(ClientKind kind)
            {
                // Arrange
                using var client = CreateDefaultAndInit(kind);
                var configRequests = 0;
                client.OnConfigSyncRequested += () => configRequests++;

                // Act
                client.SetMediaSource("media_source", "campaign_name", "group_name");

                // Assert
                configRequests.Should().Be(1);
            }

            [Test]
            [UnitTestTarget(typeof(MagifyClient), nameof(MagifyClient.OnConfigLoaded))]
            // When went to background and return to foreground BEFORE session interval
            [TestCase(0, 2, 0, ClientKind.Full)]
            [TestCase(0, 2, 0, ClientKind.Minimal)]
            [TestCase(1, 10, 0, ClientKind.Full)]
            [TestCase(1, 10, 0, ClientKind.Minimal)]
            // When went to background and return to foreground AFTER session interval
            [TestCase(2, 1, 1, ClientKind.Full)]
            [TestCase(2, 1, 1, ClientKind.Minimal)]
            [TestCase(5, 2, 1, ClientKind.Full)]
            [TestCase(5, 2, 1, ClientKind.Minimal)]
            public async Task WhenWentToBackground_AndReturnToForeground_ThenCheckOnConfigLoaded(int delaySeconds, int sessionInterval, int expectedCallbacks, ClientKind kind)
            {
                // Arrange
                using var client = CreateDefaultAndInit(kind);
                var platform = (PlatformDefault)GetPlatformAPI(client);
                GetSessionCounter(client).Interval.Value = TimeSpan.FromSeconds(sessionInterval);
                var onConfigLoadedCalls = 0;
                client.OnConfigSyncRequested += () => onConfigLoadedCalls++;

                // Act
                platform.GoToBackground();
                if (delaySeconds > 0) await UniTask.WaitForSeconds(delaySeconds);
                platform.ReturnToForeground();

                // Assert
                onConfigLoadedCalls.Should().Be(expectedCallbacks);
            }

            [Test]
            [TestCase(ClientKind.Full)]
            [TestCase(ClientKind.Minimal)]
            public async Task WhenLocaleChanged_ThenOnConfigSyncRequestedCalled(ClientKind kind)
            {
                // Arrange
                using var client = CreateDefaultAndInit(kind);
                var configRequests = 0;
                client.OnConfigSyncRequested += () => configRequests++;
                using var _ = new TemporaryEnableMagifyLogger();
                LoggingScope.IterateScopes().ForEach(MagifyLogger.EnableTag);

                // Act
                ((PlatformDefault)GetPlatformAPI(client)).TweakUserLocale("rr-RR");
                try
                {
                    await UniTask.WaitUntil(() => configRequests > 0, cancellationToken: GetCancellationToken());
                }
                catch (Exception e)
                {
                    Debug.LogError($"Exception caught: {e.Message}");
                }

                // Assert
                configRequests.Should().Be(1);
            }

            [Test]
            [TestCase(ClientKind.Full)]
            [TestCase(ClientKind.Minimal)]
            public async Task WhenConfigLoaded_ThenChangeScopesAndSyncAgain_ThenSuccess(ClientKind kind)
            {
                // Arrange
                var network = new NetworkMoq((_, message, _) => NetworkMoq.OkResult(message));
                using var client = CreateDefault(kind: kind, networkClient: network);
                var configRequests = 0;
                client.OnConfigSyncRequested += () => configRequests++;
                var scopeRequested = false;
                client.OnConfigLoaded += () =>
                {
                    if (scopeRequested) return;
                    scopeRequested = true;
                    client.RemoteContextScopes.Value = ConfigScope.Limits;
                    client.Sync();
                };

                // Act
                try
                {
                    client.InitializeSdk();
                    client.Environment.Value = Environment.Staging;
                    client.Setup();
                    await TaskHelper.WaitUntil(() => scopeRequested, token: GetCancellationToken());
                }
                catch (Exception e)
                {
                    Assert.Fail(e.Message);
                }

                // Assert
                configRequests.Should().Be(2);
            }
        }
    }
}