using System.Collections.Generic;
using FluentAssertions;
using NUnit.Framework;
using static Magify.CounterKey.Flags;

namespace Magify.Tests
{
    internal class CounterKeyTests
    {
        [Test]
        public void WhenCounterKeyCreated_AndAllPropertiesUsedInMethod_ThenAllPropertiesAreCorrect()
        {
            // Arrange
            var key = CounterKey.GetKey(
                campaignName: "campaignName",
                campaignType: CampaignType.Bonus,
                nestedName: "nestedName",
                nestedType: "nestedType",
                source: "source",
                sourceType: SourceType.Event,
                limitId: 12);

            // Act

            // Assert
            key.CampaignName.Should().Be("campaignName");
            key.CampaignType.Should().Be(CampaignType.Bonus);
            key.NestedName.Should().Be("nestedName");
            key.NestedType.Should().Be("nestedType");
            key.Source.Should().Be("source");
            key.SourceType.Should().Be(SourceType.Event);
            key.LimitId.Should().Be(12);
        }

        private static List<(CounterKey Key, string ExpectedString, CounterKey.Flags ExpectedMask)> KeysToStringInputs { get; } = new()
        {
            (CounterKey.GetKey(campaignName: "campaignName"), "c=campaignName", CampaignNameProp),
            (CounterKey.GetKey(campaignType: CampaignType.Subscription), "ct=subscription", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.Interstitial), "ct=interstitial", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.RateReview), "ct=rate_and_review", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.External), "ct=external", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.CrossPromo), "ct=cross_promo", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.Notification), "ct=notification", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.RewardedVideo), "ct=rewarded_video", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.Banner), "ct=banner", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.InApp), "ct=inapp", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoInApp), "ct=limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoSubscription), "ct=subscription_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoExternal), "ct=external_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoCrossPromo), "ct=cross_promo_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoRewardedVideo), "ct=rewarded_video_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.Bonus), "ct=bonus_inapp", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoBonus), "ct=bonus_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.Mixed), "ct=mixed", CampaignTypeProp),
            (CounterKey.GetKey(campaignType: CampaignType.LtoMixed), "ct=mixed_limited_time_offer", CampaignTypeProp),
            (CounterKey.GetKey(nestedName: "nested"), "n=nested", NestedNameProp),
            (CounterKey.GetKey(nestedType: "nestedType"), "nt=nestedType", NestedTypeProp),
            (CounterKey.GetKey(source: "source"), "s=source", SourceProp),
            (CounterKey.GetKey(sourceType: SourceType.Event), "st=event", SourceTypeProp),
            (CounterKey.GetKey(sourceType: SourceType.Trigger), "st=trigger", SourceTypeProp),
            (CounterKey.GetKey(sourceType: SourceType.Placement), "st=placement", SourceTypeProp),
            (CounterKey.GetKey(limitId: 10L), "l=10", LimitIdProp),
            (CounterKey.GetKey(
                    campaignName: "campaignName",
                    campaignType: CampaignType.Banner,
                    nestedName: "nested",
                    nestedType: "nestedType",
                    source: "source",
                    sourceType: SourceType.Event,
                    limitId: 10L),
                "c=campaignNameЎct=bannerЎn=nestedЎnt=nestedTypeЎs=sourceЎst=eventЎl=10",
                CampaignNameProp | CampaignTypeProp | NestedNameProp | NestedTypeProp | SourceProp | SourceTypeProp | LimitIdProp),
        };

        [TestCaseSource(nameof(KeysToStringInputs))]
        public void WhenUsedOnlySpecificProperties_AndConvertedToString_ThenStringIsCorrect((CounterKey Key, string ExpectedString, CounterKey.Flags) input)
        {
            // Arrange
            var (key, expectedString, _) = input;

            // Act
            var keyAsString = key.ToString();

            // Assert
            keyAsString.Should().Be(expectedString);
        }

        [TestCaseSource(nameof(KeysToStringInputs))]
        public void WhenUsedOnlySpecificProperties_AndGettingUsedProperties_ThenMaskIsCorrect((CounterKey Key, string, CounterKey.Flags ExpectedFlags) input)
        {
            // Arrange
            var (key, _, expectedFlags) = input;

            // Act
            var usedProperties = key.GetUsedProperties();

            // Assert
            usedProperties.Should().Be(expectedFlags);
        }

        [Test]
        public void WhenCounterKeyConvertedToString_AndParsedBackToKey_ThenAllPropertiesAreSame()
        {
            // Arrange
            var key1 = CounterKey.GetKey(
                campaignName: "campaignName",
                campaignType: CampaignType.Bonus,
                nestedName: "nestedName",
                nestedType: "nestedType",
                source: "source",
                sourceType: SourceType.Event,
                limitId: 12);

            // Act
            var key2 = CounterKey.FromString(key1.ToString());

            // Assert
            key2.CampaignName.Should().Be(key1.CampaignName);
            key2.CampaignType.Should().Be(key1.CampaignType);
            key2.NestedName.Should().Be(key1.NestedName);
            key2.NestedType.Should().Be(key1.NestedType);
            key2.Source.Should().Be(key1.Source);
            key2.SourceType.Should().Be(key1.SourceType);
            key2.LimitId.Should().Be(key1.LimitId);
        }
    }
}