﻿using System.Collections.Generic;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class SegmentationsCollectionTests
        {
            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenSegmentationsShouldBeSameSegmentationsCount(ContextKind kind)
            {
                //Arrange
                var segmentationsCollection = new SegmentationsCollection();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)segmentationsCollection).UpdateContext(context, kind);

                //Assert
                segmentationsCollection.Values.Value!.Count.Should()!.Be(context.Segmentations?.Length);
                segmentationsCollection.Values.Value!.ShouldBeSameInAnyOrder(context.Segmentations);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            public void WhenUpdateContextWithDefaultContextKind_ThenSegmentationsShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                var segmentationsCollection = new SegmentationsCollection();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)segmentationsCollection).UpdateContext(context, kind);

                //Assert
                segmentationsCollection.Values.Value.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWithEmptySegmentations_ThenSegmentationsShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                var segmentationsCollection = new SegmentationsCollection();
                var context = new CampaignsContext();

                //Act
                ((IContextListener)segmentationsCollection).UpdateContext(context, kind);

                //Assert
                segmentationsCollection.Values.Value.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextMultiplyTimes_ThenUpdatedSegmentationsShouldBeSameAsContext(ContextKind kind)
            {
                //Arrange
                var segmentationsCollection = new SegmentationsCollection();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)segmentationsCollection).UpdateContext(firstContext, kind);
                var firstSegmentations = segmentationsCollection.Values.Value;
                ((IContextListener)segmentationsCollection).UpdateContext(secondContext, kind);
                var secondSegmentations = segmentationsCollection.Values.Value;

                //Assert
                firstSegmentations!.ShouldBeSameInAnyOrder(firstContext.Segmentations);
                secondSegmentations!.ShouldBeSameInAnyOrder(secondContext.Segmentations);
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                var segmentationsCollection = new SegmentationsCollection();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                IReadOnlyList<string> firstSegmentations = default;
                IReadOnlyList<string> secondSegmentations = default;

                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)segmentationsCollection).UpdateContext(firstContext, kind);
                    firstSegmentations = segmentationsCollection.Values.Value;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)segmentationsCollection).UpdateContext(secondContext, kind);
                    secondSegmentations = segmentationsCollection.Values.Value;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstSegmentations.Count.Should()!.Be(firstContext.Segmentations?.Length);
                secondSegmentations.Count.Should()!.Be(secondContext.Segmentations?.Length);

                firstSegmentations!.ShouldBeSameInAnyOrder(firstContext.Segmentations);
                secondSegmentations!.ShouldBeSameInAnyOrder(secondContext.Segmentations);
            }
        }
    }
}