﻿using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class LimitsHolderTests
        {
            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenLimitsShouldBeNotNull(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var limitsHolder, out var _);
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)limitsHolder).UpdateContext(context, kind);

                //Assert
                limitsHolder.Limits.Should()!.NotBeNull();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWithEmptyLimits_ThenLimitsShouldBeNull(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var limitsHolder, out var _);
                var context = new CampaignsContext();

                //Act
                ((IContextListener)limitsHolder).UpdateContext(context, kind);

                //Assert
                limitsHolder.Limits.Should()!.BeNull();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenResetLimits_ThenLimitsShouldBeNull(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var limitsHolder, out var _);
                var context = new CampaignsContext();

                //Act
                ((IContextListener)limitsHolder).UpdateContext(context, kind);
                limitsHolder.Reset();

                //Assert
                limitsHolder.Limits.Should()!.BeNull();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenIterateTemporaryLimitsForProduct_ThenTemporaryLimitsShouldBeSameCountAsInContext(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var limitsHolder, out var _);
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)limitsHolder).UpdateContext(context, kind);
                var temporaryLimits = limitsHolder.IterateTemporaryLimitsForProductWithLock("product");

                //Assert
                temporaryLimits.TemporaryLimits.Count().Should()!.Be(context.Limits?.TempLimits?.Count);
            }

            [Test]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextMultiplyTimes_ThenUpdatedLimitsShouldBeSameAsContext(ContextKind kind)
            {
                //Arrange
                using var _ = CreateSubsystems(out var limitsHolder, out var _);
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)limitsHolder).UpdateContext(firstContext, kind);
                var firstUpdateLimits = limitsHolder.Limits;
                ((IContextListener)limitsHolder).UpdateContext(secondContext, kind);
                var secondUpdateLimits = limitsHolder.Limits;

                //Assert
                firstUpdateLimits.Should()!.Be(firstContext.Limits);
                secondUpdateLimits.Should()!.Be(secondContext.Limits);
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                using var listener = CreateSubsystems(out var limitsHolder, out var _);
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                Model.Limits firstUpdateLimits = default;
                Model.Limits secondUpdateLimits = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)limitsHolder).UpdateContext(firstContext, kind);
                    firstUpdateLimits = firstContext.Limits;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)limitsHolder).UpdateContext(secondContext, kind);
                    secondUpdateLimits = secondContext.Limits;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                firstUpdateLimits.Should()!.Be(firstContext.Limits);
                secondUpdateLimits.Should()!.Be(secondContext.Limits);
            }
        }
    }
}