﻿using System.Collections.Generic;
using System.Linq;
using System.Threading;
using System.Threading.Tasks;
using Cysharp.Threading.Tasks;
using FluentAssertions;
using Magify.Model;
using NUnit.Framework;

namespace Magify.Tests
{
    internal partial class ContextListenerTests
    {
        internal class CampaignsCollectionTests
        {
            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContext_ThenCampaignRecordShouldBeSameCampaignModelsCount(ContextKind kind)
            {
                //Arrange
                var listener = new CampaignsCollection();
                var context = CreateCampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(context, kind);

                //Assert
                var campaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;
                campaigns.Count.Should()!.Be(context.CampaignModels?.Count);
                campaigns.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(context.CampaignModels?.Select(c => c?.Name).ToArray());
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextWithEmptyCampaignsContext_ThenResultShouldBeEmpty(ContextKind kind)
            {
                //Arrange
                var listener = new CampaignsCollection();
                var context = new CampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(context, kind);

                //Assert
                var campaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;
                campaigns.Should()!.BeEmpty();
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public void WhenUpdateContextMultiplyTimes_ThenUpdatedCampaignsShouldBeSameAsContext(ContextKind kind)
            {
                //Arrange
                var listener = new CampaignsCollection();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();

                //Act
                ((IContextListener)listener).UpdateContext(firstContext, kind);
                var firstUpdateCampaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;
                ((IContextListener)listener).UpdateContext(secondContext, kind);
                var secondUpdateCampaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;

                //Assert
                firstContext.Should()!.NotBe(secondContext);
                firstUpdateCampaigns.Count.Should()!.Be(firstContext.CampaignModels?.Count);
                firstUpdateCampaigns.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.CampaignModels?.Select(c => c?.Name).ToArray());

                secondUpdateCampaigns.Count.Should()!.Be(secondContext.CampaignModels?.Count);
                secondUpdateCampaigns.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.CampaignModels?.Select(c => c?.Name).ToArray());
            }

            [Test]
            [TestCase(ContextKind.Default)]
            [TestCase(ContextKind.Saved)]
            [TestCase(ContextKind.Downloaded)]
            public async Task WhenUpdateContextOnThreadPool_ThenUpdateContextShouldBeThreadSafety(ContextKind kind)
            {
                //Arrange
                var listener = new CampaignsCollection();
                var firstContext = CreateCampaignsContext();
                var secondContext = CreateCampaignsContext();
                IReadOnlyList<CampaignRecord> firstUpdateCampaigns = default;
                IReadOnlyList<CampaignRecord> secondUpdateCampaigns = default;
                var barrier = new Barrier(2);

                //Act
                var firstTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(firstContext, kind);
                    firstUpdateCampaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;
                });

                var secondTask = UniTask.RunOnThreadPool(() =>
                {
                    barrier.SignalAndWait();
                    ((IContextListener)listener).UpdateContext(secondContext, kind);
                    secondUpdateCampaigns = kind == ContextKind.Default ? listener.DefaultCampaigns : listener.CurrentCampaigns;
                });

                await UniTask.WhenAll(firstTask, secondTask);

                //Assert
                //firstContext.Should()!.NotBe(secondContext);
                firstUpdateCampaigns.Count.Should()!.Be(firstContext.CampaignModels?.Count);
                firstUpdateCampaigns.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(firstContext.CampaignModels?.Select(c => c?.Name).ToArray());

                secondUpdateCampaigns.Count.Should()!.Be(secondContext.CampaignModels?.Count);
                secondUpdateCampaigns.Select(c => c?.Name).ToArray()
                    .ShouldBeSameInAnyOrder(secondContext.CampaignModels?.Select(c => c?.Name).ToArray());
            }
        }
    }
}